﻿/*
 * Copyright (C)  2013  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Drawing;
using System.Reflection;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace plexdata.Utilities
{
    public class FixedSingleBorderRenderer : NativeWindow
    {
        private Control control = null;

        public FixedSingleBorderRenderer(Control control, Color border)
            : base()
        {
            if (control == null) { throw new ArgumentNullException("control"); }

            this.control = control;
            this.control.HandleCreated += new EventHandler(this.OnParentHandleCreated);
            this.control.HandleDestroyed += new EventHandler(this.OnParentHandleDestroyed);

            this.BorderColor = border;
        }

        public FixedSingleBorderRenderer(Control control)
            : this(control, SystemColors.ControlDark)
        {
        }

        public Color BorderColor { get; set; }

        protected override void WndProc(ref Message msg)
        {
            // For more details about this hack see link below.
            // http://social.msdn.microsoft.com/Forums/en-US/windowsuidevelopment/thread/a407591a-4b1e-4adc-ab0b-3c8b3aec3153/

            const int WM_NCPAINT = 0x85;

            if (msg.Msg == WM_NCPAINT && IsFixedSingleBorder())
            {
                IntPtr hWnd = msg.HWnd;
                IntPtr hClip = msg.WParam;
                IntPtr hTemp = IntPtr.Zero;
                IntPtr hDC = GetDCEx(hWnd, IntPtr.Zero, DCX_WINDOW | DCX_USESTYLE);

                RECT rc = new RECT();
                GetClientRect(hWnd, ref rc);

                // NOTE: Using a text box control cause that the border is not painted. 
                //       This might be because of such a control does not set its non-client 
                //       area accordingly or for any other reason.
                RECT rw = new RECT();
                GetWindowRect(hWnd, ref rw);

                // Save rectangle offset.
                int x = rw.left;
                int y = rw.top;

                MapWindowPoints(IntPtr.Zero, hWnd, ref rw, 2);

                OffsetRect(ref rc, -rw.left, -rw.top);
                OffsetRect(ref rw, -rw.left, -rw.top);

                if (hClip == (IntPtr)0 || hClip == (IntPtr)1)
                {
                    ExcludeClipRect(hDC, rc.left, rc.top, rc.right, rc.bottom);
                    hTemp = IntPtr.Zero;
                }
                else
                {
                    hTemp = CreateRectRgn(rc.left + x, rc.top + y, rc.right + x, rc.bottom + y);

                    if (NULLREGION == CombineRgn(hTemp, hClip, hTemp, RGN_DIFF))
                    {
                        // Nothing to paint...
                        return;
                    }
                    else
                    {
                        OffsetRgn(hTemp, -x, -y);
                        ExtSelectClipRgn(hDC, hTemp, RGN_AND);
                    }
                }

                using (Graphics graphics = Graphics.FromHdc(hDC))
                {
                    // Use this instead of DrawRectangle() 
                    // because it reduces flickering!
                    graphics.Clear(this.BorderColor);
                }

                ReleaseDC(hWnd, hDC);
                if (hTemp != IntPtr.Zero)
                {
                    DeleteObject(hTemp);
                }
            }
            else
            {
                base.WndProc(ref msg);
            }
        }

        private void OnParentHandleCreated(object sender, EventArgs args)
        {
            this.AssignHandle(((Control)sender).Handle);
        }

        private void OnParentHandleDestroyed(object sender, EventArgs args)
        {
            this.ReleaseHandle();
        }

        private bool IsFixedSingleBorder()
        {
            const int GWL_STYLE = (-16);
            const int WS_BORDER = 0x800000;

            if ((GetWindowLong(this.Handle, GWL_STYLE) & WS_BORDER) != 0)
            {
                return true;
            }
            else
            {
                // This piece of code is only needed if the parent control supports border 
                // style but it doesn't  support WS_BORDER style. Usually WS_BORDER is set. 
                PropertyInfo info = this.control.GetType().GetProperty("BorderStyle");
                if (info != null)
                {
                    return ((BorderStyle)info.GetValue(this.control, null)) == BorderStyle.FixedSingle;
                }
                else
                {
                    return false;
                }
            }
        }

        #region Win32 access declarations.

        private const int DCX_WINDOW = 0x00000001;
        private const int DCX_CACHE = 0x00000002;
        private const int DCX_NORESETATTRS = 0x00000004;
        private const int DCX_CLIPCHILDREN = 0x00000008;
        private const int DCX_CLIPSIBLINGS = 0x00000010;
        private const int DCX_PARENTCLIP = 0x00000020;
        private const int DCX_EXCLUDERGN = 0x00000040;
        private const int DCX_INTERSECTRGN = 0x00000080;
        private const int DCX_EXCLUDEUPDATE = 0x00000100;
        private const int DCX_INTERSECTUPDATE = 0x00000200;
        private const int DCX_LOCKWINDOWUPDATE = 0x00000400;
        // http://www.catch22.net/tuts/win32-tips-tricks
        // DCX_USESTYLE: Undocumented, something related to WM_NCPAINT message.
        private const int DCX_USESTYLE = 0x00010000;
        private const int DCX_VALIDATE = 0x00200000;

        private const int RGN_AND = 1;
        private const int RGN_OR = 2;
        private const int RGN_XOR = 3;
        private const int RGN_DIFF = 4;
        private const int RGN_COPY = 5;
        private const int RGN_MIN = RGN_AND;
        private const int RGN_MAX = RGN_COPY;

        private const int ERROR = 0;
        private const int NULLREGION = 1;
        private const int SIMPLEREGION = 2;
        private const int COMPLEXREGION = 3;

        private struct RECT
        {
            public int left;
            public int top;
            public int right;
            public int bottom;
        }

        [DllImport("user32.dll", SetLastError = true)]
        static extern int GetWindowLong(IntPtr hWnd, int nIndex);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern IntPtr GetDCEx(IntPtr hWnd, IntPtr hRgnClip, int flags);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern int ReleaseDC(IntPtr hWnd, IntPtr hDC);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool GetClientRect(IntPtr hWnd, ref RECT lpRect);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool GetWindowRect(IntPtr hWnd, ref RECT lpRect);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern int MapWindowPoints(IntPtr hWndFrom, IntPtr hWndTo, ref RECT rect, uint count);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern int MapWindowPoints(IntPtr hWndFrom, IntPtr hWndTo, ref Rectangle rect, uint count);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool OffsetRect(ref RECT lprc, int dx, int dy);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern int ExcludeClipRect(IntPtr hDC, int nLeftRect, int nTopRect, int nRightRect, int nBottomRect);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern IntPtr CreateRectRgn(int nLeftRect, int nTopRect, int nRightRect, int nBottomRect);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern int CombineRgn(IntPtr hrgnDest, IntPtr hrgnSrc1, IntPtr hrgnSrc2, int fnCombineMode);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern int OffsetRgn(IntPtr hrgn, int nXOffset, int nYOffset);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern int ExtSelectClipRgn(IntPtr hdc, IntPtr hrgn, int fnMode);

        [DllImport("gdi32.dll", SetLastError = true)]
        private static extern bool DeleteObject(IntPtr hObject);

        #endregion //Win32 access declarations.
    }
}
