﻿/*
 * Copyright (C)  2013  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace plexdata.Utilities
{
    public class ClipboardNotification : NativeWindow
    {
        public event EventHandler<EventArgs> ClipboardChanged;

        private IntPtr nextViewer = IntPtr.Zero;

        public ClipboardNotification(Control parent)
        {
            if (parent == null)
            {
                throw new ArgumentNullException("parent");
            }

            parent.HandleCreated += new EventHandler(this.OnHandleCreated);
            parent.HandleDestroyed += new EventHandler(this.OnHandleDestroyed);
        }

        private void OnHandleCreated(object sender, EventArgs args)
        {
            Control control = sender as Control;
            if (control != null)
            {
                this.AssignHandle(control.Handle);
                this.nextViewer = SetClipboardViewer(control.Handle);
            }
        }

        private void OnHandleDestroyed(object sender, EventArgs args)
        {
            Control control = sender as Control;
            if (control != null)
            {
                ChangeClipboardChain(control.Handle, this.nextViewer);
            }
            this.ReleaseHandle();
        }

        private void OnClipboardChanged()
        {
            if (this.ClipboardChanged != null)
            {
                this.ClipboardChanged(this, EventArgs.Empty);
            }
        }

        protected override void WndProc(ref Message message)
        {
            const int WM_DRAWCLIPBOARD = 0x308;
            const int WM_CHANGECBCHAIN = 0x030D;

            switch (message.Msg)
            {
                case WM_DRAWCLIPBOARD:
                    {
                        this.OnClipboardChanged();
                        SendMessage(this.nextViewer, message.Msg, message.WParam, message.LParam);
                        break;
                    }
                case WM_CHANGECBCHAIN:
                    {
                        if (message.WParam == this.nextViewer)
                        {
                            this.nextViewer = message.LParam;
                        }
                        else
                        {
                            SendMessage(this.nextViewer, message.Msg, message.WParam, message.LParam);
                        }
                        break;
                    }

                default:
                    {
                        base.WndProc(ref message);
                        break;
                    }
            }
        }

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern IntPtr SetClipboardViewer(IntPtr hWndNewViewer);

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern bool ChangeClipboardChain(IntPtr hWndRemove, IntPtr hWndNewNext);

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern int SendMessage(IntPtr hwnd, int wMsg, IntPtr wParam, IntPtr lParam);
    }
}
