﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;
using System.Collections.Generic;

namespace plexdata.ImageViewer
{
    public class SlideshowHelper
    {
        private int interval = 0;
        private int upper = 0;
        private int lower = 0;

        private bool repeated = false;
        private bool suspended = false;
        private Timer timer = null;
        private ImageBoxEx display = null;
        private MaximizeHandler maximizer = null;

        private LinkedListNode<FileInfo> slide = null;
        private LinkedList<FileInfo> slides = null;

        public SlideshowHelper(MainForm parent)
            : base()
        {
            this.Parent = parent;
        }

        private static ComponentResourceManager resourceManager = null;
        public static ComponentResourceManager ResourceManager
        {
            get
            {
                if (SlideshowHelper.resourceManager == null)
                {
                    SlideshowHelper.resourceManager = new ComponentResourceManager(typeof(SlideshowHelper));
                }
                return SlideshowHelper.resourceManager;
            }
        }

        public MainForm Parent { get; private set; }

        public bool IsActive { get; private set; }

        public bool Start(NavigatorNode selectedNode)
        {
            try
            {
                if (!this.IsActive && this.Parent != null && this.InitializeSlides(selectedNode))
                {
                    // Hide all controls.
                    foreach (Control current in this.Parent.Controls)
                    {
                        current.Visible = false;
                    }

                    // Create and initialize slideshow display.
                    this.display = new ImageBoxEx();
                    this.Parent.Controls.Add(this.display);

                    this.display.Dock = DockStyle.Fill;
                    this.display.BorderStyle = BorderStyle.None;
                    this.display.BackColor = this.Parent.Settings.Slideshow.BackColor;
                    this.display.Padding = new Padding(this.Parent.Settings.Slideshow.ImagePadding);
                    this.display.Interpolation = this.Parent.Settings.Slideshow.Interpolation;
                    this.display.SuppressScrollbars = true;

                    this.display.Focus();

                    // Create and initialize contextmenu.
                    this.InitializeMenu();

                    // Show parent form maximized.
                    this.maximizer = new MaximizeHandler(this.Parent);
                    if (this.maximizer.Maximize())
                    {
                        // Last but not least start slideshow.
                        this.suspended = this.Parent.Settings.Slideshow.Suspended;
                        this.repeated = this.Parent.Settings.Slideshow.Repeated;
                        this.interval = this.Parent.Settings.Slideshow.Interval;
                        this.upper = this.interval * 2;
                        this.lower = this.interval / 2;
                        this.timer = new Timer();
                        this.timer.Interval = 1; // Start immediately!
                        this.timer.Tick += this.SlideshowTimerTick;

                        if (!this.suspended)
                        {
                            this.timer.Start();
                        }
                        else
                        {
                            // Flickers a little bit...
                            this.LoadNextImage();
                        }

                        this.IsActive = true;
                    }
                    else
                    {
                        this.Stop();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                this.Stop();
            }
            return this.IsActive;
        }

        public void Stop()
        {
            try
            {
                if (this.Parent != null)
                {
                    // Save current settings.
                    this.Parent.Settings.Slideshow.Suspended = this.suspended;
                    this.Parent.Settings.Slideshow.Repeated = this.repeated;

                    if (this.timer != null)
                    {
                        // Destroy slide timer.
                        this.timer.Stop();
                        this.timer.Dispose();
                        this.timer = null;
                    }

                    if (this.display != null)
                    {
                        // Save current interpolation value.
                        this.Parent.Settings.Slideshow.Interpolation = this.display.Interpolation;

                        // Now destroy slide display.
                        if (this.display.ContextMenu != null)
                        {
                            this.display.ContextMenu.Dispose();
                        }
                        this.Parent.Controls.Remove(this.display);
                        this.display.Dispose();
                        this.display = null;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                // Ensure that the parent window is restored and 
                // terminate application in case of an exception.
                try
                {
                    if (this.maximizer != null)
                    {
                        this.maximizer.Restore();
                        this.maximizer = null;
                    }

                    // Show all hidden controls.
                    foreach (Control current in this.Parent.Controls)
                    {
                        if (!current.Visible)
                        {
                            current.Visible = true;
                        }
                    }
                }
                catch (Exception exception)
                {
                    MessageBox.Show(
                        exception.Message + "\n\n" + exception.ToString(),
                        Properties.Resources.MESSAGE_CAPTION_ERROR,
                        MessageBoxButtons.OK, MessageBoxIcon.Error);

                    Application.Exit();
                }

                this.IsActive = false;
            }
        }

        private void SlideshowTimerTick(object sender, EventArgs args)
        {
            this.LoadNextImage();

            if (this.IsActive)
            {
                // Update context menu items!
                this.ContextMenuPopupHandler(this.display.ContextMenu, EventArgs.Empty);
            }
        }

        private bool InitializeSlides(NavigatorNode node)
        {
            this.slide = null;
            this.slides = new LinkedList<FileInfo>();
            if (node != null)
            {
                if (node.Parent != null)
                {
                    foreach (NavigatorNode current in node.Parent.Nodes)
                    {
                        if (current.IsFileNode)
                        {
                            FileInfo info = current.NodeInfo<FileInfo>();
                            if (info != null)
                            {
                                this.slides.AddLast(info);
                            }
                        }
                    }
                }
            }
            return (this.slides.Count > 0);
        }

        private void InitializeMenu()
        {
            this.display.ContextMenu = new ContextMenu(new MenuItem[]{
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_SUSPEND"), this.OnMenuItemSuspend),
                new MenuItem("-"),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_FORWARD"), this.OnMenuItemForward),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_BACKWARD"), this.OnMenuItemBackward),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_REPEATED"), this.OnMenuItemRepeated),
                new MenuItem("-"),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_FASTER"), this.OnMenuItemFaster),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_SLOWER"), this.OnMenuItemSlower),
                new MenuItem("-"),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_SMOOTH"), this.OnMenuItemSmooth),
                new MenuItem("-"),
                new MenuItem(SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_FINISH"), this.OnMenuItemFinish),
            });
            this.display.ContextMenu.Popup += this.ContextMenuPopupHandler;
        }

        private void LoadNextImage()
        {
            try
            {
                if (this.slide == null)
                {
                    this.slide = this.slides.First;
                }
                else
                {
                    if (this.slide == this.slides.Last)
                    {
                        if (!this.repeated)
                        {
                            this.Stop();
                            return;
                        }
                        else
                        {
                            this.slide = this.slides.First;
                        }
                    }
                    else
                    {
                        this.slide = this.slide.Next;
                    }
                }

                if (!this.suspended)
                {
                    this.timer.Stop();
                    this.timer.Interval = this.interval;
                }

                // Load next slide image.
                FileInfo fileInfo = this.slide.Value;
                using (FileStream fileStream = fileInfo.OpenRead())
                {
                    if (Program.MainForm.Settings.General.AutoRotate)
                    {
                        this.display.Image = RotateImageHelper.RotateFlip(Image.FromStream(fileStream));
                    }
                    else
                    {
                        this.display.Image = Image.FromStream(fileStream);
                    }

                    this.display.FitToSize();
                }

                if (!this.suspended)
                {
                    this.timer.Start();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                this.Stop();
            }
        }

        private void LoadPrevImage()
        {
            try
            {
                if (this.slide == null)
                {
                    // This is really unexpected because at least the first 
                    // slide should be valid after the timer has started.
                    throw new ArgumentOutOfRangeException("slide");
                }
                else
                {
                    if (this.slide == this.slides.First)
                    {
                        if (!this.repeated)
                        {
                            this.Stop();
                            return;
                        }
                        else
                        {
                            this.slide = this.slides.Last;
                        }
                    }
                    else
                    {
                        this.slide = this.slide.Previous;
                    }
                }

                if (!this.suspended)
                {
                    this.timer.Stop();
                    this.timer.Interval = this.interval;
                }

                // Load next slide image.
                FileInfo fileInfo = this.slide.Value;
                using (FileStream fileStream = fileInfo.OpenRead())
                {
                    if (Program.MainForm.Settings.General.AutoRotate)
                    {
                        this.display.Image = RotateImageHelper.RotateFlip(Image.FromStream(fileStream));
                    }
                    else
                    {
                        this.display.Image = Image.FromStream(fileStream);
                    }

                    this.display.FitToSize();
                }

                if (!this.suspended)
                {
                    this.timer.Start();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                this.Stop();
            }
        }

        #region Context menu and menu item event handler implementation.

        private void ContextMenuPopupHandler(object sender, EventArgs args)
        {
            try
            {
                if (sender is ContextMenu)
                {
                    foreach (MenuItem current in (sender as ContextMenu).MenuItems)
                    {
                        if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_SUSPEND")) == 0)
                        {
                            current.Checked = this.suspended;
                        }
                        else if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_REPEATED")) == 0)
                        {
                            current.Checked = this.repeated;
                        }
                        else if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_SMOOTH")) == 0)
                        {
                            current.Checked = this.display.Interpolation;
                        }
                        else if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_FASTER")) == 0)
                        {
                            current.Enabled = (this.interval > this.lower);
                        }
                        else if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_SLOWER")) == 0)
                        {
                            current.Enabled = (this.interval < this.upper);
                        }
                        else if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_FORWARD")) == 0)
                        {
                            current.Enabled = this.repeated ? true : (this.slide != this.slides.Last);
                        }
                        else if (String.Compare(current.Text, SlideshowHelper.ResourceManager.GetString("MENUITEMTEXT_BACKWARD")) == 0)
                        {
                            current.Enabled = this.repeated ? true : (this.slide != this.slides.First);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnMenuItemForward(object sender, EventArgs args)
        {
            this.LoadNextImage();
        }

        private void OnMenuItemBackward(object sender, EventArgs args)
        {
            this.LoadPrevImage();
        }

        private void OnMenuItemFinish(object sender, EventArgs args)
        {
            this.Stop();
        }

        private void OnMenuItemSuspend(object sender, EventArgs args)
        {
            try
            {
                if (this.timer != null)
                {
                    if (this.suspended)
                    {
                        this.suspended = false;
                        this.LoadNextImage();
                        this.timer.Start();
                    }
                    else
                    {
                        this.suspended = true;
                        this.timer.Stop();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnMenuItemFaster(object sender, EventArgs args)
        {
            try
            {
                if (this.timer != null && this.interval > this.lower)
                {
                    this.timer.Stop();
                    this.interval = Math.Max(this.interval / 2, this.lower);
                    this.timer.Interval = this.interval;
                    this.timer.Start();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnMenuItemSlower(object sender, EventArgs args)
        {
            try
            {
                if (this.timer != null && this.interval < this.upper)
                {
                    this.timer.Stop();
                    this.interval = Math.Min(this.interval * 2, this.upper);
                    this.timer.Interval = this.interval;
                    this.timer.Start();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnMenuItemRepeated(object sender, EventArgs args)
        {
            this.repeated = !this.repeated;
        }

        private void OnMenuItemSmooth(object sender, EventArgs args)
        {
            try
            {
                this.display.Interpolation = !this.display.Interpolation;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        #endregion // Context menu and menu item event handler implementation.
    }
}
