﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Text;
using System.Drawing;
using Microsoft.Win32;
using System.Reflection;
using System.Diagnostics;
using System.Windows.Forms;
using System.Drawing.Imaging;
using System.Security.Principal;
using System.Runtime.InteropServices;

namespace plexdata.ImageViewer
{
    internal class ShellExtension
    {
        private const string CLASSID = "$ClassID$";
        private const string NAME = "$Name$";
        private const string DESCRIPTION = "$Description$";
        private const string LABEL = "$Label$";
        private const string EXECUTABLE = "$Executable$";
        private const string HELPSTRING = "$HelpString$";
        private const string ICONDATA = "$IconData$";

        public ShellExtension()
            : base()
        {
        }

        public bool IsUserAdministrator()
        {
            try
            {
                // Check role of currently logged-on user.
                WindowsIdentity user = WindowsIdentity.GetCurrent();
                WindowsPrincipal principal = new WindowsPrincipal(user);
                return principal.IsInRole(WindowsBuiltInRole.Administrator);
            }
            catch (UnauthorizedAccessException)
            {
                return false;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return false;
            }
        }

        public bool Is64BitPlatform()
        {
            try
            {
                // Check underlying processor architecture.
                SystemInfo info = new SystemInfo();
                ShellExtension.GetNativeSystemInfo(out info);
                return (info.processorArchitecture == ShellExtension.PROCESSOR_ARCHITECTURE_AMD64);
            }
            catch
            {
                return false;
            }
        }

        public bool Register()
        {
            string cfgFile = String.Empty;
            string dllFile = String.Empty;
            if (this.PrepareConfiguration(out cfgFile))
            {
                if (this.ExtractBinary(out dllFile))
                {
                    return this.Register(dllFile, cfgFile);
                }
            }
            return false;
        }

        public bool Unregister()
        {
            string dllFile = String.Empty;
            if (this.ExtractBinary(out dllFile))
            {
                // Unregister only requires the Class ID!
                return this.Unregister(dllFile, this.ClassID);
            }
            else
            {
                return false;
            }
        }

        #region Private method implementation.

        private bool Register(string dllFile, string cfgFile)
        {
            try
            {
                if (this.IsUserAdministrator())
                {
                    using (Process rundll = new Process())
                    {
                        rundll.StartInfo.FileName = "rundll32.exe";
                        rundll.StartInfo.Arguments = String.Format("\"{0}\",RegisterExtension {1}", dllFile, cfgFile);
                        rundll.Start();
                        rundll.WaitForExit();

                        try // Try to delete unneeded configuration file.
                        {
                            FileInfo file = new FileInfo(cfgFile);
                            file.Delete();
                        }
                        catch (Exception exception)
                        {
                            Debug.WriteLine(exception);
                        }
                    }
                    return true;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private bool Unregister(string dllFile, string cfgFile)
        {
            try
            {
                if (this.IsUserAdministrator())
                {
                    using (Process rundll = new Process())
                    {
                        rundll.StartInfo.FileName = "rundll32.exe";
                        rundll.StartInfo.Arguments = String.Format("\"{0}\",UnregisterExtension {1}", dllFile, cfgFile);
                        rundll.Start();
                        rundll.WaitForExit();
                    }
                    return true;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private bool PrepareConfiguration(out string filename)
        {
            filename = String.Empty;

            try
            {
                filename = Path.Combine(Path.GetDirectoryName(Path.GetFullPath(
                    Assembly.GetExecutingAssembly().Location)), "pdcmse.config.xml");

                // Load pre-configured registration data from resources and 
                // replace all pre-configured placeholders with real data.
                string config = global::plexdata.ImageViewer.Properties.Resources.pdcmseConfig
                    .Replace(ShellExtension.CLASSID, this.ClassID)
                    .Replace(ShellExtension.NAME, this.Name)
                    .Replace(ShellExtension.DESCRIPTION, this.Description)
                    .Replace(ShellExtension.LABEL, this.Label)
                    .Replace(ShellExtension.EXECUTABLE, this.Executable)
                    .Replace(ShellExtension.HELPSTRING, this.HelpString)
                    .Replace(ShellExtension.ICONDATA, this.IconData);

                FileInfo file = new FileInfo(filename);
                using (StreamWriter stream = file.CreateText())
                {
                    stream.Write(config);
                }
                return true;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private bool ExtractBinary(out string filename)
        {
            filename = String.Empty;

            try
            {
                filename = Path.GetDirectoryName(
                    Path.GetFullPath(Assembly.GetExecutingAssembly().Location));

                // Load platform dependent binary resource.
                byte[] data = null;
                if (this.Is64BitPlatform())
                {
                    filename = Path.Combine(filename, "pdcmse64.dll");
                    data = global::plexdata.ImageViewer.Properties.Resources.pdcmse64;
                }
                else
                {
                    filename = Path.Combine(filename, "pdcmse32.dll");
                    data = global::plexdata.ImageViewer.Properties.Resources.pdcmse32;
                }

                if (data != null)
                {
                    FileInfo file = new FileInfo(filename);
                    if (!file.Exists)
                    {
                        // Do not create binary if already exists!
                        using (FileStream stream = file.Create())
                        {
                            stream.Write(data, 0, data.Length);
                        }
                    }
                    return true;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        public bool IsRegistered()
        {
            try
            {
                // REMARK: Copied from Shell Extension Composer!
                string INSTALLED_EXTENSIONS = "SOFTWARE\\plexdata\\pdcmse\\InstalledExtensions";
                using (RegistryKey root = Registry.LocalMachine.OpenSubKey(INSTALLED_EXTENSIONS))
                {
                    Guid guid = new Guid(this.ClassID);
                    foreach (string current in root.GetValueNames())
                    {
                        if (guid.CompareTo(new Guid(current)) == 0)
                        {
                            return true;
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        #endregion // Private method implementation.

        #region Private property implementation.

        private string ClassID
        {
            get
            {
                try
                {
                    object[] result = Assembly
                        .GetExecutingAssembly()
                        .GetCustomAttributes(typeof(GuidAttribute), false);

                    return ((GuidAttribute)result[0]).Value;
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }

                return String.Empty;
            }
        }

        private string Name
        {
            get
            {
                try
                {
                    object[] result = Assembly
                        .GetExecutingAssembly()
                        .GetCustomAttributes(typeof(AssemblyTitleAttribute), false);

                    return ((AssemblyTitleAttribute)result[0]).Title;
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }

                return String.Empty;
            }
        }

        private string Description
        {
            get
            {
                try
                {
                    object[] result = Assembly
                        .GetExecutingAssembly()
                        .GetCustomAttributes(typeof(AssemblyDescriptionAttribute), false);

                    return ((AssemblyDescriptionAttribute)result[0]).Description;
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }

                return String.Empty;
            }
        }

        private string Label
        {
            get { return String.Format(Properties.Resources.SHELL_EXTENSION_LABEL, this.Name); }
        }

        private string Executable
        {
            get
            {
                return Path.GetFullPath(Assembly.GetExecutingAssembly().Location);
            }
        }

        private string HelpString
        {
            get { return String.Format(Properties.Resources.SHELL_EXTENSION_HELP, this.Name); }
        }

        private string IconData
        {
            get
            {
                try
                {
                    Bitmap bitmap = Program.ShellIcon;

                    // Check if current OS version is Vista or higher.
                    if (Environment.OSVersion.Version.Major >= 6)
                    {
                        bitmap = (new Icon(
                            global::plexdata.ImageViewer.Properties.Resources.MainIcon,
                            new Size(16, 16))).ToBitmap();
                    }

                    MemoryStream stream = new MemoryStream();
                    bitmap.Save(stream, ImageFormat.Bmp);
                    byte[] array = stream.ToArray();

                    StringBuilder result = new StringBuilder(array.Length * 2);
                    foreach (byte current in array)
                    {
                        result.AppendFormat("{0:X2}", current);
                    }
                    return result.ToString();
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }
                return String.Empty;
            }
        }

        #endregion // Private property implementation.

        #region Native system function and constant declarations.

        private const ushort PROCESSOR_ARCHITECTURE_INTEL = 0x000; // x86
        private const ushort PROCESSOR_ARCHITECTURE_AMD64 = 0x0009; // x64 (AMD or Intel)
        private const ushort PROCESSOR_ARCHITECTURE_IA64 = 0x0006; // Intel Itanium Processor Family (IPF)
        private const ushort PROCESSOR_ARCHITECTURE_UNKNOWN = 0xFFFF; // Unknown processor.

        [StructLayout(LayoutKind.Sequential)]
        private struct SystemInfo
        {
            public ushort processorArchitecture;
            public ushort reserved;
            public uint pageSize;
            public IntPtr minApplicationAddress;
            public IntPtr maxApplicationAddress;
            public uint activeProcessorMask;
            public uint numberOfProcessors;
            public uint processorType;
            public uint allocationGranularity;
            public ushort processorLevel;
            public ushort processorRevision;
        }

        [DllImport("kernel32.dll", SetLastError = true, CallingConvention = CallingConvention.StdCall)]
        private static extern void GetNativeSystemInfo(out SystemInfo systemInfo);

        #endregion // Native system function and constant declarations.
    }
}
