﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Text;
using System.Diagnostics;
using System.Runtime.InteropServices;

namespace plexdata.ImageViewer
{
    internal static class ShellAccess
    {
        #region Shell Execute dependent declarations and implementations.

        public static bool ShowPropertiesDialog(string filename)
        {
            SHELLEXECUTEINFO info = new SHELLEXECUTEINFO();
            info.cbSize = System.Runtime.InteropServices.Marshal.SizeOf(info);
            info.lpVerb = "properties";
            info.lpFile = filename;
            info.nShow = SW_SHOW;
            info.fMask = SEE_MASK_INVOKEIDLIST;
            return ShellExecuteEx(ref info);
        }

        private const int SW_SHOW = 5;
        private const uint SEE_MASK_INVOKEIDLIST = 0x0000000C;

        [DllImport("shell32.dll", CharSet = CharSet.Auto)]
        private static extern bool ShellExecuteEx(ref SHELLEXECUTEINFO lpExecInfo);

        [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Auto)]
        private struct SHELLEXECUTEINFO
        {
            public int cbSize;
            public uint fMask;
            public IntPtr hwnd;
            [MarshalAs(UnmanagedType.LPTStr)]
            public string lpVerb;
            [MarshalAs(UnmanagedType.LPTStr)]
            public string lpFile;
            [MarshalAs(UnmanagedType.LPTStr)]
            public string lpParameters;
            [MarshalAs(UnmanagedType.LPTStr)]
            public string lpDirectory;
            public int nShow;
            public IntPtr hInstApp;
            public IntPtr lpIDList;
            [MarshalAs(UnmanagedType.LPTStr)]
            public string lpClass;
            public IntPtr hkeyClass;
            public uint dwHotKey;
            public IntPtr hIcon;
            public IntPtr hProcess;
        }

        #endregion // Shell Execute dependent declarations and implementations.

        #region Shell Folder dependent declarations and implementations.

        public static string GetDisplayName(int csidl)
        {
            string result = String.Empty;
            IntPtr pidl = IntPtr.Zero;

            try
            {
                csidl |= CSIDL_FLAG_NO_ALIAS;

                SHFILEINFO sfi = new SHFILEINFO();
                int size = Marshal.SizeOf(sfi);
                int flags = SHGFI_PIDL | SHGFI_DISPLAYNAME | SHGFI_TYPENAME;

                if (Succeeded(SHGetFolderLocation(IntPtr.Zero, csidl, IntPtr.Zero, 0, out pidl)))
                {
                    SHGetFileInfo(pidl, -1, ref sfi, size, flags);

                    result = sfi.displayName;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                Marshal.FreeCoTaskMem(pidl);
            }

            return result;
        }

        public static string GetFolderPath(int csidl)
        {
            string result = String.Empty;

            try
            {
                csidl |= CSIDL_FLAG_NO_ALIAS;

                StringBuilder builder = new StringBuilder(MAX_PATH);

                if (Succeeded(SHGetFolderPath(IntPtr.Zero, csidl, IntPtr.Zero, 0, builder)))
                {
                    result = builder.ToString();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }

            return result;
        }

        private static bool Succeeded(IntPtr hResult)
        {
            return (hResult != null && hResult.ToInt32() == 0);
        }

        [StructLayout(LayoutKind.Sequential)]
        private struct SHFILEINFO
        {
            public IntPtr hIcon;
            public Int32 iIcon;
            public UInt32 attributes;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = MAX_PATH)]
            public string displayName;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 80)]
            public string typeName;
        };

        private const int MAX_PATH = 260;

        private const int SHGFI_PIDL = 0x000000008;
        private const int SHGFI_DISPLAYNAME = 0x000000200;              // get display name
        private const int SHGFI_TYPENAME = 0x000000400;                 // get type name

        private const int CSIDL_FLAG_NO_ALIAS = 0x00001000;             // combine with CSIDL_ value to insure non-alias versions of the pidl

        public const int CSIDL_DESKTOP = 0x00000000;                    // <desktop>
        public const int CSIDL_DRIVES = 0x00000011;                     // <my computer>
        public const int CSIDL_FAVORITES = 0x00000006;                  // <user name>\Favorites
        public const int CSIDL_COMMON_FAVORITES = 0x0000001f;           // <all users>\Favorites
        public const int CSIDL_MYPICTURES = 0x00000027;                 // <user name>\My Pictures
        public const int CSIDL_COMMON_PICTURES = 0x00000036;            // <all users>\Pictures
        public const int CSIDL_DESKTOPDIRECTORY = 0x00000010;           // <user name>\Desktop
        public const int CSIDL_COMMON_DESKTOPDIRECTORY = 0x00000019;    // <all users>\Desktop

        [DllImport("shell32.dll")]
        private static extern IntPtr SHGetFileInfo(IntPtr pData, int attributes, ref SHFILEINFO sfi, int size, int flags);

        [DllImport("shell32.dll")]
        private static extern IntPtr SHGetFolderLocation(IntPtr hWnd, int csidl, IntPtr hToken, int flags, out IntPtr pData);

        [DllImport("shell32.dll")]
        private static extern IntPtr SHGetFolderPath(IntPtr hWnd, int csidl, IntPtr hToken, int flags, StringBuilder path);

        #endregion // Shell Folder dependent declarations and implementations.
    }
}
