﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Xml;
using System.Linq;
using System.Drawing;
using System.Reflection;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;
using System.Xml.Serialization;
using System.Collections.Generic;

namespace plexdata.ImageViewer
{
    public class Settings : ICloneable
    {
        #region Class construction.

        public Settings()
            : base()
        {
            this.Maintain = new Maintain();
            this.General = new General();
            this.Zooming = new Zooming();
            this.Slideshow = new Slideshow();
            this.Filtering = new Filtering();
        }

        public Settings(Settings other)
            : this()
        {
            this.Maintain = new Maintain(other.Maintain);
            this.General = new General(other.General);
            this.Zooming = new Zooming(other.Zooming);
            this.Slideshow = new Slideshow(other.Slideshow);
            this.Filtering = new Filtering(other.Filtering);
        }

        #endregion // Class construction.

        #region Public property implementation.

        public Maintain Maintain { get; set; }

        public General General { get; set; }

        public Zooming Zooming { get; set; }

        public Slideshow Slideshow { get; set; }

        public Filtering Filtering { get; set; }

        #endregion // Public property implementation.

        #region Public static implementation.

        public static bool Save(string filename, Settings root)
        {
            bool success = false;
            XmlSerializer serializer = null;
            TextWriter writer = null;

            try
            {
                serializer = new XmlSerializer(typeof(Settings));
                writer = new StreamWriter(filename);
                serializer.Serialize(writer, root);

                success = true;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                if (writer != null) { writer.Close(); }
                writer = null;
                serializer = null;
            }
            return success;
        }

        public static bool Load(string filename, out Settings root)
        {
            bool success = false;
            XmlSerializer serializer = null;
            TextReader reader = null;

            root = default(Settings);

            try
            {
                serializer = new XmlSerializer(typeof(Settings));
                reader = new StreamReader(filename);
                root = (Settings)serializer.Deserialize(reader);

                if (root != default(Settings))
                {
                    root.EnsureScreenLocation();
                    success = true;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                if (reader != null) { reader.Close(); }
                reader = null;
                serializer = null;
            }
            return success;
        }

        public static string Filename
        {
            get
            {
                string path = Path.GetFullPath(Assembly.GetExecutingAssembly().Location);

                return
                    Path.GetDirectoryName(path) +
                    Path.DirectorySeparatorChar +
                    Path.GetFileNameWithoutExtension(path) +
                    ".xml";
            }
        }

        #endregion // Public static implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Settings(this);
        }

        #endregion // ICloneable member implementation.

        #region Private method implementation.

        private void EnsureScreenLocation()
        {
            Rectangle bounds = Screen.PrimaryScreen.WorkingArea;
            Point location = this.Maintain.Location;
            Size size = this.Maintain.Size;

            foreach (Screen screen in Screen.AllScreens)
            {
                if (screen.Bounds.Contains(new Rectangle(location, size)))
                {
                    return; // Location is still within bounds.
                }
            }

            // Adjust X location to be on the primary screen!

            int x = 0;
            if (location.X < bounds.Left)
            {
                x = bounds.Left;
            }
            else if (location.X + size.Width > bounds.Left + bounds.Right)
            {
                x = bounds.Right - size.Width;
            }
            else
            {
                x = location.X;
            }

            // Adjust Y location to be on the primary screen!

            int y = 0;
            if (location.Y < bounds.Top)
            {
                y = bounds.Top;
            }
            else if (location.Y + size.Height > bounds.Top + bounds.Bottom)
            {
                y = bounds.Bottom - size.Height;
            }
            else
            {
                y = location.Y;
            }

            this.Maintain.Location = new Point(x, y);
        }

        #endregion // Private method implementation.
    }

    public class Maintain : ICloneable
    {
        // NOTE: Put all non editable settings here!

        #region Class construction.

        public Maintain()
            : base()
        {
            this.Size = new Size(640, 480);
            this.Location = new Point(100, 100);
            this.WindowState = FormWindowState.Normal;
            this.MainToolbar = new Point(0, 0);
            this.ZoomToolbar = new Point(0, 0);
            this.NavigatorWidth = 200;
            this.NavigatorVisible = true;
            this.DisplayHeight = 280;
            this.NotesVisible = true;
            this.ActiveSettings = String.Empty;
            this.Favorites = null;
        }

        public Maintain(Maintain other)
            : this()
        {
            this.Size = new Size(other.Size.Width, other.Size.Height);
            this.Location = new Point(other.Location.X, other.Location.Y);
            this.WindowState = other.WindowState;
            this.ZoomToolbar = new Point(other.ZoomToolbar.X, other.ZoomToolbar.Y);
            this.MainToolbar = new Point(other.MainToolbar.X, other.MainToolbar.Y);
            this.NavigatorWidth = other.NavigatorWidth;
            this.NavigatorVisible = other.NavigatorVisible;
            this.DisplayHeight = other.DisplayHeight;
            this.NotesVisible = other.NotesVisible;
            this.ActiveSettings = other.ActiveSettings;
            this.Favorites = other.Favorites;
        }

        #endregion // Class construction.

        #region Public property implementation.

        public Size Size { get; set; }

        public Point Location { get; set; }

        public FormWindowState WindowState { get; set; }

        public Point MainToolbar { get; set; }

        public Point ZoomToolbar { get; set; }

        public int NavigatorWidth { get; set; }

        public bool NavigatorVisible { get; set; }

        public int DisplayHeight { get; set; }

        public bool NotesVisible { get; set; }

        // Last active settings page.
        public string ActiveSettings { get; set; }

        // Do not use type "List<string>" because otherwise 
        // "someone" in the XML parser appends items instead 
        // of overwriting complete list content!
        private List<Favorite> favorites = null;
        public Favorite[] Favorites
        {
            get { return this.favorites.ToArray(); }
            set
            {
                this.favorites = new List<Favorite>();
                if (value != null)
                {
                    foreach (Favorite favorite in value)
                    {
                        this.favorites.Add(new Favorite(favorite));
                    }
                }
            }
        }

        #endregion // Public property implementation.

        #region Public method implementation.

        public bool HasFavorite(DirectoryInfo directory)
        {
            if (directory != null)
            {
                return this.HasFavorite(new Favorite(directory));
            }
            else
            {
                return false;
            }
        }

        public bool HasFavorite(Favorite favorite)
        {
            return this.favorites.Contains(favorite);
        }

        public bool AppendFavorite(DirectoryInfo directory)
        {
            if (directory != null)
            {
                return this.AppendFavorite(new Favorite(directory));
            }
            else
            {
                return false;
            }
        }

        public bool AppendFavorite(Favorite favorite)
        {
            if (!this.HasFavorite(favorite))
            {
                this.favorites.Add(new Favorite(favorite));
                return true;
            }
            else
            {
                // An existing item is already added, isn't it?
                return false;
            }
        }

        public bool RemoveFavorite(DirectoryInfo directory)
        {
            if (directory != null)
            {
                return this.RemoveFavorite(new Favorite(directory));
            }
            else
            {
                return false;
            }
        }

        public bool RemoveFavorite(Favorite favorite)
        {
            if (this.HasFavorite(favorite))
            {
                return this.favorites.Remove(favorite);
            }
            else
            {
                // Non existing item is removed already, isn't it?
                return true;
            }
        }

        #endregion // Public method implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Maintain(this);
        }

        #endregion // ICloneable member implementation.
    }

    public class Favorite : ICloneable, IEquatable<Favorite>
    {
        #region Class construction.

        public Favorite()
            : base()
        {
            this.Name = String.Empty;
            this.Folder = String.Empty;
        }

        public Favorite(string name, string folder)
            : this()
        {
            this.Name = new string(name.ToCharArray());
            this.Folder = new string(folder.ToCharArray());
        }

        public Favorite(DirectoryInfo directory)
            : this(directory.Name, directory.FullName)
        {
        }

        public Favorite(Favorite other)
            : this(other.Name, other.Folder)
        {
        }

        #endregion // Class construction.

        #region Public property implementation.

        public string Name { get; set; }

        public string Folder { get; set; }

        #endregion // Public property implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Favorite(this);
        }

        #endregion // ICloneable member implementation.

        #region IEquatable<Favorite> member implementation.

        public bool Equals(Favorite other)
        {
            if (other == null)
            {
                return false;
            }
            else
            {
                return
                    (0 == String.Compare(this.Name, other.Name, true)) &&
                    (0 == String.Compare(this.Folder, other.Folder, true));
            }
        }

        #endregion // IEquatable<Favorite> member implementation.
    }

    public class General : ICloneable
    {
        #region Class construction.

        public General()
            : base()
        {
            this.Font = SystemFonts.DefaultFont;
            this.ForeColor = SystemColors.WindowText;
            this.BackColor = SystemColors.Window;
            this.AutoAdjust = true;
            this.AutoRotate = true;
        }

        public General(General other)
            : this()
        {
            this.Font = (Font)other.Font.Clone();
            this.ForeColor = Color.FromArgb(other.ForeColorARGB);
            this.BackColor = Color.FromArgb(other.BackColorARGB);
            this.AutoAdjust = other.AutoAdjust;
            this.AutoRotate = other.AutoRotate;
        }

        #endregion // Class construction.

        #region Public property implementation.

        [XmlIgnore]
        public Font Font { get; set; }

        [XmlElement("Font")]
        public string FontSTR
        {
            get
            {
                try
                {
                    if (this.Font != null)
                    {
                        TypeConverter converter = TypeDescriptor.GetConverter(typeof(Font));
                        return converter.ConvertToString(this.Font);
                    }
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }
                return String.Empty;
            }
            set
            {
                try
                {
                    TypeConverter converter = TypeDescriptor.GetConverter(typeof(Font));
                    this.Font = (Font)converter.ConvertFromString(value);
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }
            }
        }

        [XmlIgnore]
        public Color ForeColor { get; set; }

        [XmlElement("ForeColor")]
        public int ForeColorARGB
        {
            get { return this.ForeColor.ToArgb(); }
            set { this.ForeColor = Color.FromArgb(value); }
        }

        [XmlIgnore]
        public Color BackColor { get; set; }

        [XmlElement("BackColor")]
        public int BackColorARGB
        {
            get { return this.BackColor.ToArgb(); }
            set { this.BackColor = Color.FromArgb(value); }
        }

        public bool AutoAdjust { get; set; }

        public bool AutoRotate { get; set; }

        #endregion // Public property implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new General(this);
        }

        #endregion // ICloneable member implementation.
    }

    public class Zooming : ICloneable
    {
        #region Class construction.

        public Zooming()
            : base()
        {
            this.ErrorText = Properties.Resources.ZOOMING_ERROR_TEXT;
            this.Interpolation = true;
            this.ImagePadding = 10;
            this.DefaultFactor = 100;
            this.MinimumFactor = 10;
            this.MaximumFactor = 6400;
        }

        public Zooming(Zooming other)
            : this()
        {
            this.ErrorText = new string(other.ErrorText.ToCharArray());
            this.Interpolation = other.Interpolation;
            this.ImagePadding = other.ImagePadding;
            this.DefaultFactor = other.DefaultFactor;
            this.MinimumFactor = other.MinimumFactor;
            this.MaximumFactor = other.MaximumFactor;
        }

        #endregion // Class construction.

        #region Public property implementation.

        public string ErrorText { get; set; }

        public bool Interpolation { get; set; }

        public int ImagePadding { get; set; }

        public int DefaultFactor { get; set; }

        public int MinimumFactor { get; set; }

        public int MaximumFactor { get; set; }

        #endregion // Public property implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Zooming(this);
        }

        #endregion // ICloneable member implementation.
    }

    public class Slideshow : ICloneable
    {
        #region Class construction.

        public Slideshow()
            : base()
        {
            this.BackColor = Color.Black;
            this.Suspended = false;
            this.Repeated = false;
            this.Interval = 5000;
            this.Interpolation = true;
            this.ImagePadding = 10;
        }

        public Slideshow(Slideshow other)
            : this()
        {
            this.BackColor = Color.FromArgb(other.BackColorARGB);
            this.Suspended = other.Suspended;
            this.Repeated = other.Repeated;
            this.Interval = other.Interval;
            this.Interpolation = other.Interpolation;
            this.ImagePadding = other.ImagePadding;
        }

        #endregion // Class construction.

        #region Public property implementation.

        [XmlIgnore]
        public Color BackColor { get; set; }

        [XmlElement("BackColor")]
        public int BackColorARGB
        {
            get { return this.BackColor.ToArgb(); }
            set { this.BackColor = Color.FromArgb(value); }
        }

        public bool Suspended { get; set; }

        public bool Repeated { get; set; }

        public int Interval { get; set; }

        public bool Interpolation { get; set; }

        public int ImagePadding { get; set; }

        #endregion // Public property implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Slideshow(this);
        }

        #endregion // ICloneable member implementation.
    }

    public class Filtering : ICloneable
    {
        private List<Format> formats = new List<Format>();

        #region Initialize private default format list.

        private static List<Format> defaults = new List<Format>(
            new Format[] {
                new Format("PNG",  true,  "Format: Portable Network Graphics (*.png)",
                    new string[] { ".png" }),
                new Format("BMP",  true,  "Format: Bitmap (*.bmp)",
                    new string[] { ".bmp" }),
                new Format("JPEG", true,  "Format: Joint Photographic Experts Group (*.jpg, *.jpeg, *.jpe, *.jif, *.jfif, *jfi)",
                    new string[] { ".jpg", ".jpeg", ".jpe", ".jif", ".jfif", ".jfi" }),
                new Format("GIF",  true,  "Format: Graphics Interchange Format (.gif)",
                    new string[] { ".gif" }),
                new Format("ICON", true, "Format: Symbol (*.ico)",
                    new string[] { ".ico" }),
                new Format("TIFF", false, "Format: Tagged Image File Format (*.tiff, *.tif)",
                    new string[] { ".tiff", ".tif" }),
                new Format("META", false, "Format: Windows Metafile (*.wmf, *.emf)",
                    new string[] { ".wmf", ".emf" }),
        });

        #endregion // Initialize private default format list

        #region Class construction.

        public Filtering()
            : base()
        {
            // Just use a copy of default formats as initial value.
            this.formats = new List<Format>(Filtering.defaults);
        }

        public Filtering(Filtering other)
            : this()
        {
            // A copy constructor just takes over given data 
            // and does not change any of the values.
            this.formats = new List<Format>();
            foreach (Format format in other.Formats)
            {
                this.formats.Add(new Format(format));
            }
        }

        #endregion // Class construction.

        #region Public property implementation.

        // Do not use type "List<Format>" because otherwise 
        // "someone" in the XML parser appends items instead 
        // of overwriting complete list content!
        public Format[] Formats
        {
            get { return this.formats.ToArray(); }
            set
            {
                this.formats = new List<Format>();
                if (value != null)
                {
                    foreach (Format current in value)
                    {
                        this.formats.Add(new Format(
                            current.Name, current.Enabled,
                            this.GetDisplay(current.Name),
                            this.GetExtensions(current.Name)));
                    }
                }
            }
        }

        #endregion // Public property implementation.

        #region Public method implementation.

        public List<string> GetSearchPattern()
        {
            return this.GetSearchPattern(true);
        }

        public List<string> GetSearchPattern(bool enabled)
        {
            List<string> result = new List<string>();
            foreach (Format current in this.Formats)
            {
                // All enabled search pattern are requested!
                if (enabled && current.Enabled)
                {
                    result.AddRange(current.GetSearchPattern());
                }
                // All disabled search pattern are requested!
                else if (!enabled && !current.Enabled)
                {
                    result.AddRange(current.GetSearchPattern());
                }
            }
            return result;
        }

        #endregion // Public method implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Filtering(this);
        }

        #endregion // ICloneable member implementation.

        #region Private method implementation.

        private string GetDisplay(string name)
        {
            try
            {
                return (Filtering.defaults.First(item => item.Name == name)).Display;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return String.Empty;
            }
        }

        private string[] GetExtensions(string name)
        {
            try
            {
                return (Filtering.defaults.First(item => item.Name == name)).Extensions;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return new string[] { };
            }
        }

        #endregion // Private method implementation.
    }

    public class Format : ICloneable
    {
        #region Class construction.

        public Format()
            : base()
        {
            this.Name = String.Empty;
            this.Enabled = false;
            this.Display = String.Empty;
            this.Extensions = new string[] { };
        }

        public Format(string name, bool enabled, string display, string[] extensions)
            : this()
        {
            this.Name = new string(name.ToCharArray());
            this.Enabled = enabled;
            this.Display = new string(display.ToCharArray());
            this.Extensions = (new List<string>(extensions)).ToArray();
        }

        public Format(Format other)
            : this(other.Name, other.Enabled, other.Display, other.Extensions)
        {
        }

        #endregion // Class construction.

        #region Public property implementation.

        [XmlAttribute]
        public string Name { get; set; }

        [XmlAttribute]
        public bool Enabled { get; set; }

        [XmlIgnore]
        public string Display { get; set; }

        [XmlIgnore]
        public string[] Extensions { get; set; }

        #endregion // Public property implementation.

        #region Public method implementation.

        public List<string> GetSearchPattern()
        {
            List<string> result = new List<string>();
            foreach (string current in this.Extensions)
            {
                // Remove spaces and wildcards, if any.
                result.Add(current.Replace(" ", "").Replace("*", "").Replace("?", ""));
            }
            return result;
        }

        #endregion // Public method implementation.

        #region ICloneable member implementation.

        public object Clone()
        {
            return new Format(this);
        }

        #endregion // ICloneable member implementation.
    }
}
