﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Threading;
using System.Reflection;
using System.Diagnostics;
using System.Globalization;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace plexdata.ImageViewer
{
    static class Program
    {
        /// <summary>
        /// The main entry point for the application.
        /// </summary>
        [STAThread]
        static void Main()
        {
            try
            {
                Application.EnableVisualStyles();
                Application.SetCompatibleTextRenderingDefault(false);

                Program.SplashScreen.Show();
                Application.DoEvents();

                Application.Run(Program.MainForm);
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private static MainForm mainForm = null;
        internal static MainForm MainForm
        {
            get
            {
                if (Program.mainForm == null)
                {
                    Program.mainForm = new MainForm();
                }
                return Program.mainForm;
            }
        }

        private static SplashScreen splashScreen = null;
        internal static SplashScreen SplashScreen
        {
            get
            {
                if (Program.splashScreen == null)
                {
                    Program.splashScreen = new SplashScreen();
                }
                return Program.splashScreen;
            }
        }

        internal static string ApplicationTitle
        {
            get { return AboutBox.Title; }
        }

        internal static string ApplicationVersion
        {
            get { return AboutBox.Version; }
        }

        internal static string ApplicationCopyright
        {
            get { return AboutBox.Copyright; }
        }

        internal static Icon ApplicationIcon
        {
            get { return global::plexdata.ImageViewer.Properties.Resources.MainIcon; }
        }

        internal static Icon SettingsIcon
        {
            get { return Icon.FromHandle(global::plexdata.ImageViewer.Properties.Resources.SettingsIcon.GetHicon()); }
        }

        internal static Icon ClipboardIcon
        {
            get { return Icon.FromHandle(global::plexdata.ImageViewer.Properties.Resources.ClipboardIcon.GetHicon()); }
        }

        internal static Icon InfoIcon
        {
            get { return Icon.FromHandle(global::plexdata.ImageViewer.Properties.Resources.InfoIcon.GetHicon()); }
        }

        internal static Bitmap ShellIcon
        {
            get { return global::plexdata.ImageViewer.Properties.Resources.ShellIcon; }
        }

        internal static void ShowHelp()
        {
            try
            {
                string help = String.Empty;
                string path = Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location);
                string file = Path.GetFileNameWithoutExtension(Assembly.GetExecutingAssembly().Location);
                CultureInfo info = Thread.CurrentThread.CurrentUICulture;

                // Rules to find the help file.
                // * The neutral (en) help file has to be put into same folder where 
                //   the executable is located.
                // * Language depending help files has to be put into their belonging 
                //   culture folder, e.g. ".\de-DE\" for "German (Germany)" help files 
                //   or  ".\de\" for "German (Neutral)" help files.
                // * The names of all help files are identical!
                //
                // Test sequence:
                // --------------
                // 1) The existence of the help file for the current culture (e.g. "de-DE") 
                //    is checked.
                // 2) The existence of the help file for current culture's parent (e.g. "de") 
                //    is checked.
                // 3) The existence of the neutral help file is checked.

                // Prepare the help file's name as described above.
                file = String.Format("{0}.chm", file);

                // Check for help file's path as described above.
                if (info != null)
                {
                    // Create a path like ".\de-DE\<help>.chm" 
                    // and check if this help file exists.
                    help = Path.Combine(Path.Combine(path, info.Name), file);
                    if (!File.Exists(help))
                    {
                        if (info.Parent != null)
                        {
                            // Create a path like ".\de\<help>.chm" 
                            // and check if this help file exists.
                            help = Path.Combine(Path.Combine(path, info.Parent.Name), file);
                            if (!File.Exists(help))
                            {
                                // Use the executable's path instead.
                                help = Path.Combine(path, file);
                            }
                        }
                        else
                        {
                            return; // Error case!
                        }
                    }
                }
                else
                {
                    return; // Error case!
                }

                // Works find but the help window stays on top in z-order!
                // Help.ShowHelp(null, help, HelpNavigator.TableOfContents);

                // Works fine too but the help window does not terminate
                // automatically when the appliation is closed.
                // Process help = new Process();
                // help.StartInfo.FileName = help;
                // help.Start();

                // Nevertheless, each solution above is not perfect! So we need to 
                // call HtmlHelp() function directly because it solves all problems!
                Program.HtmlHelp(IntPtr.Zero, help, Program.HH_DISPLAY_TOC, IntPtr.Zero);
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        internal static bool InSearchPattern(string extension, List<string> pattern)
        {
            try
            {
                foreach (string current in pattern)
                {
                    if (0 == String.Compare(current, extension, true))
                    {
                        return true;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        #region HtmlHelp import stuff.

        // See file "htmlhelp.h" and http://support.microsoft.com/kb/317406

        private const uint HH_DISPLAY_TOPIC = 0x0000;
        private const uint HH_HELP_FINDER = 0x0000;
        private const uint HH_DISPLAY_TOC = 0x0001;
        private const uint HH_DISPLAY_INDEX = 0x0002;
        private const uint HH_DISPLAY_SEARCH = 0x0003;

        [DllImport("hhctrl.ocx", CharSet = CharSet.Auto)]
        private static extern IntPtr HtmlHelp(IntPtr hCaller, String helpfile, uint command, IntPtr data);

        #endregion // HtmlHelp import stuff.
    }
}
