/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace plexdata.ImageViewer
{
    internal class MaximizeHandler
    {
        #region Private member declaration.

        private Form target = null;
        private Rectangle bounds = default(Rectangle);
        private FormWindowState state = FormWindowState.Normal;
        private FormBorderStyle style = FormBorderStyle.None;

        #endregion // Private member declaration.

        public MaximizeHandler(Form target)
            : base()
        {
            this.target = target;
        }

        #region Public property implementation.

        public bool IsMaximized { get; private set; }

        #endregion // Public property implementation.

        #region Public method implementation.

        public bool Maximize()
        {
            try
            {
                if (!this.IsMaximized)
                {
                    // Save current settings.
                    this.state = this.target.WindowState;
                    this.style = this.target.FormBorderStyle;
                    this.bounds = this.target.Bounds;

                    // Set new settings.
                    this.target.WindowState = FormWindowState.Maximized;
                    this.target.FormBorderStyle = FormBorderStyle.None;

                    // Try to maximize current target window.
                    this.IsMaximized = this.TryMaximize(this.target.Handle);
                }
                return true;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return false;
            }
        }

        public bool Restore()
        {
            try
            {
                if (this.IsMaximized)
                {
                    this.target.WindowState = this.state;
                    this.target.FormBorderStyle = this.style;
                    this.target.Bounds = this.bounds;
                    this.IsMaximized = false;
                }
                return true;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return false;
            }
        }

        #endregion // Public method implementation.

        #region Private method implementation.

        private bool TryMaximize(IntPtr hWnd)
        {
            try
            {
                return MaximizeHandler.SetWindowPos(
                    hWnd, MaximizeHandler.HWND_TOP, 0, 0,
                    MaximizeHandler.GetSystemMetrics(MaximizeHandler.SM_CXSCREEN),
                    MaximizeHandler.GetSystemMetrics(MaximizeHandler.SM_CYSCREEN),
                    MaximizeHandler.SWP_SHOWWINDOW);
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return false;
            }
        }

        #endregion // Private method implementation.

        #region Win32 API access declarations.

        [DllImport("user32.dll", SetLastError = true)]
        private static extern int GetSystemMetrics(int nIndex);

        [DllImport("user32.dll", SetLastError = true)]
        private static extern bool SetWindowPos(IntPtr hWnd, IntPtr hWndInsertAfter, int x, int y, int cx, int cy, int flags);

        private static IntPtr HWND_TOP = new IntPtr(0);

        private const int SM_CXSCREEN = 0;
        private const int SM_CYSCREEN = 1;

        private const int SWP_SHOWWINDOW = 0x0040;

        #endregion // Win32 API access declarations.
    }
}
