﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Collections.Generic;

using plexdata.Utilities;

namespace plexdata.ImageViewer
{
    public partial class MainForm : Form
    {
        private Settings settings = new Settings();
        private MaximizeHandler maximizer = null;
        private SlideshowHelper slideshow = null;
        private ClipboardNotification clipboard = null;

        private FixedSingleBorderRenderer panel1Renderer = null;
        private FixedSingleBorderRenderer panel2Renderer = null;
        private FixedSingleBorderRenderer panel3Renderer = null;

        public MainForm()
            : base()
        {
            this.InitializeComponent();

            this.maximizer = new MaximizeHandler(this);

            this.Text = Program.ApplicationTitle;
            this.Icon = Program.ApplicationIcon;

            this.panel1Renderer = new FixedSingleBorderRenderer(this.panel1Helper);
            this.panel2Renderer = new FixedSingleBorderRenderer(this.panel2Helper);
            this.panel3Renderer = new FixedSingleBorderRenderer(this.notesPanel);

            this.InitializeToolbars();
            this.InitializeTreeview();
            this.InitializeDisplay();

            this.clipboard = new ClipboardNotification(this);
            this.clipboard.ClipboardChanged += new EventHandler<EventArgs>(this.OnClipboardChanged);
        }

        public Settings Settings
        {
            get { return this.settings; }
        }

        public bool ClipboardEnabled
        {
            get
            {
                // If I did everything right, than this button's 
                // state always reflects the real clipboard state.
                return this.tbbClipboard.Enabled;
            }
        }

        protected override bool ProcessCmdKey(ref Message msg, Keys keyData)
        {
            Keys code = keyData & Keys.KeyCode;
            bool ctrl = (keyData & Keys.Control) == Keys.Control;
            bool shift = (keyData & Keys.Shift) == Keys.Shift;
            bool alt = (keyData & Keys.Alt) == Keys.Alt;

            if (this.IsSlideshow)
            {
                if (code == Keys.Escape || (!ctrl && !shift && alt && code == Keys.F4))
                {
                    // Stop slideshow...
                    this.StopSlideshow();
                }
                else if (!ctrl && !shift && !alt && code == Keys.Apps)
                {
                    return base.ProcessCmdKey(ref msg, keyData);
                }

                // Suppress any other key and key combination in slide show mode!
                return true;
            }
            else if (!ctrl && !shift && !alt)
            {
                if (code == Keys.F5 && this.CanSlideshow)
                {
                    this.StartSlideshow();
                    return true;
                }
            }
            else if (ctrl && !shift && !alt)
            {
                // [CTRL]+[T] => Toggle navigator tree...
                if (code == Keys.T)
                {
                    this.tbbNavigator.PerformClick();
                    return true;
                }
                // [CTRL]+[F] => Toggle full screen/normal window...
                else if (code == Keys.F)
                {
                    this.tbbFullscreen.PerformClick();
                    return true;
                }
                // [CTRL]+[S] => Show settings dialog...
                else if (code == Keys.S)
                {
                    this.tbbSettings.PerformClick();
                    return true;
                }
                // [CTRL]+[G] => Fit to page...
                else if (code == Keys.G)
                {
                    this.tbbFitSize.PerformClick();
                    return true;
                }
                // [CTRL]+[W] => Fit to width...
                else if (code == Keys.W)
                {
                    this.tbbFitWidth.PerformClick();
                    return true;
                }
                // [CTRL]+[L] => Rotate left (anticlockwise)...
                else if (code == Keys.L)
                {
                    this.tbbRotateLeft.PerformClick();
                    return true;
                }
                // [CTRL]+[R] => Rotate right (clockwise)...
                else if (code == Keys.R)
                {
                    this.tbbRotateRight.PerformClick();
                    return true;
                }
                // [CTRL]+[N] => Show next image...
                else if (code == Keys.N)
                {
                    this.tbbJumpNext.PerformClick();
                    return true;
                }
                // [CTRL]+[P] => Show previous image...
                else if (code == Keys.P)
                {
                    this.tbbJumpPrevious.PerformClick();
                    return true;
                }
                // [CTRL]+[+] => Zoom in...
                else if (code == Keys.Add || code == Keys.Oemplus)
                {
                    this.tbbZoomIn.PerformClick();
                    return true;
                }
                // [CTRL]+[-] => Zoom out...
                else if (code == Keys.Subtract || code == Keys.OemMinus)
                {
                    this.tbbZoomOut.PerformClick();
                    return true;
                }
                // [CTRL]+[0] => Zoom to default size...
                else if (code == Keys.D0 || code == Keys.NumPad0)
                {
                    try
                    {
                        this.imageDisplay.ZoomFactor = this.imageDisplay.DefaultZoomFactor;
                    }
                    catch (Exception exception)
                    {
                        Debug.WriteLine(exception);
                    }
                    return true;
                }
            }
            else if (ctrl && !shift && alt)
            {
                // [CTRL]+[ALT]+[A] => Show about box...
                if (code == Keys.A)
                {
                    // NOTE: Handling of shortcut CTRL+A is suppressed!
                    this.tbmAbout.PerformClick();
                    return true;
                }
                // [CTRL]+[ALT]+[C] => Show files in clipboard...
                else if (code == Keys.C)
                {
                    this.tbbClipboard.PerformClick();
                    return true;
                }
                // [CTRL]+[ALT]+[N] => Show/hide notes window...
                else if (code == Keys.N)
                {
                    this.tbbNotes.PerformClick();
                    return true;
                }
            }
            return base.ProcessCmdKey(ref msg, keyData);
        }

        protected override void OnHelpRequested(HelpEventArgs args)
        {
            this.tbmHelp.PerformClick();
            args.Handled = true;
            base.OnHelpRequested(args);
        }

        #region Slide show member implementation.

        public bool IsSlideshow { get { return (this.slideshow != null && this.slideshow.IsActive); } }

        public bool CanSlideshow { get { return (this.imageDisplay.Image != null); } }

        public void StartSlideshow()
        {
            if (this.slideshow == null)
            {
                this.slideshow = new SlideshowHelper(this);
            }
            this.slideshow.Start(this.imageNavigator.SelectedNode);
        }

        public void StopSlideshow()
        {
            if (this.slideshow != null)
            {
                this.slideshow.Stop();
            }
        }

        #endregion // Slide show member implementation.

        #region Main form event handler implementation.

        private void MainFormLoad(object sender, EventArgs args)
        {
            try
            {
                // BUG: Too much to do while loading the application. 
                //      And no better solution found! So live with it 
                //      until you find a better solution!

                // Hide main window to prevent ugly flickering. The main 
                // form is re-shown through setting the window state.
                this.Hide();

                // Do never call this function from inside the main form's constructor!
                this.UpdateMainToolbarClipboardButtonState();

                // Load persistent settings.
                Settings helper;
                if (Settings.Load(Settings.Filename, out helper))
                {
                    this.settings = new Settings(helper);
                }

                // Load defaults or current settings.
                this.LoadSettings();

                // Note that the first command line argument is always 
                // set to the path of the executable. This means that 
                // real parameters starting at the second command line 
                // argument! 
                string[] arguments = Environment.GetCommandLineArgs();
                if (arguments != null && arguments.Length > 1)
                {
                    // Such a path argument might contain unneeded double quotes.
                    string argument = arguments[1].Replace("\"", "");
                    if (this.IsValidPath(argument))
                    {
                        this.imageNavigator.ExpandToPath(argument);

                        // Show or hide navigator depending on currently 
                        // selected node type.
                        if (this.imageNavigator.SelectedNode != null)
                        {
                            if (this.imageNavigator.SelectedNode.IsFileNode)
                            {
                                // Hide navigator if shown.
                                if (!this.mainSplitter.Panel1Collapsed)
                                {
                                    this.tbbNavigator.PerformClick();
                                }
                            }
                            else
                            {
                                // Show navigator if hidden.
                                if (this.mainSplitter.Panel1Collapsed)
                                {
                                    this.tbbNavigator.PerformClick();
                                }
                            }
                        }
                    }
                }

                // Now restore last known window state.
                this.WindowState = this.settings.Maintain.WindowState;
            }
            catch (Exception exception)
            {
                MessageBox.Show(
                    exception.Message + "\n\n" + exception.ToString(),
                    Properties.Resources.MESSAGE_CAPTION_ERROR, 
                    MessageBoxButtons.OK, MessageBoxIcon.Error);
                this.Close();
            }
        }

        private void MainFormShown(object sender, EventArgs args)
        {
            // Someone deselects the "default" control! Therefore, 
            // reselect a control depending on splitter state.
            if (this.mainSplitter.Panel1Collapsed)
            {
                this.imageDisplay.Select();
            }
            else
            {
                this.imageNavigator.Select();
            }

            // Splash screen is not longer needed!
            Program.SplashScreen.Hide();

            try
            {
                // Empty current clipboard content 
                // before showing the main window.
                Clipboard.Clear();
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void MainFormLocationChanged(object sender, EventArgs args)
        {
            if (this.WindowState == FormWindowState.Normal)
            {
                this.settings.Maintain.Location = this.Location;
            }
        }

        private void MainFormSizeChanged(object sender, EventArgs args)
        {
            if (this.WindowState == FormWindowState.Normal)
            {
                this.settings.Maintain.Size = this.Size;
            }
        }

        private void MainFormClosing(object sender, FormClosingEventArgs args)
        {
            // Ensure that last edited notes are saved to file.
            this.LoadNotes(null);

            // Save last known window state right now.
            this.settings.Maintain.WindowState = this.WindowState;

            // Save last known main splitter configuration values.
            this.settings.Maintain.NavigatorVisible = !this.mainSplitter.Panel1Collapsed;
            try
            {
                // An exception may occur if window state is minimized!
                this.settings.Maintain.NavigatorWidth = this.mainSplitter.SplitterDistance;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }

            // Save last known notes splitter configuration values.
            this.settings.Maintain.NotesVisible = !this.dispSplitter.Panel2Collapsed;
            try
            {
                // An exception may occur if window state is minimized!
                this.settings.Maintain.DisplayHeight = this.dispSplitter.SplitterDistance;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }

            // Now make current settings persistent.
            Settings.Save(Settings.Filename, this.settings);
        }

        #endregion // Main form event handler implementation.

        #region Other event handler implementations.

        private void OnClipboardChanged(object sender, EventArgs args)
        {
            this.UpdateMainToolbarClipboardButtonState();
        }

        #endregion // Other event handler implementations.

        #region Private member implementation.

        private void LoadSettings()
        {
            try
            {
                // Apply all non editable settings!
                this.Size = this.settings.Maintain.Size;
                this.Location = this.settings.Maintain.Location;

                // Sometimes it seems to be that the tool strip's layout manager 
                // does what it wants. This means that the toolbars are anywhere, 
                // but not there where they should. On the other hand this kind 
                // of restoring the toolbar positions it is better then nothing...
                this.zoomToolbar.Location = this.settings.Maintain.ZoomToolbar;
                this.mainToolbar.Location = this.settings.Maintain.MainToolbar;

                try
                {   // An exception may occur if window state is minimized!
                    this.mainSplitter.SplitterDistance = this.settings.Maintain.NavigatorWidth;
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }

                if (!this.settings.Maintain.NavigatorVisible && !this.mainSplitter.Panel1Collapsed)
                {
                    this.tbbNavigator.PerformClick();
                    this.imageNavigator.SelectedNode = this.imageNavigator.Nodes.FirstNode;
                }

                try
                {   // An exception may occur if window state is minimized!
                    this.dispSplitter.SplitterDistance = this.settings.Maintain.DisplayHeight;
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }

                if (!this.settings.Maintain.NotesVisible && !this.dispSplitter.Panel2Collapsed)
                {
                    this.tbbNotes.PerformClick();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }

            // Apply all editable settings!
            this.ApplySettings();
        }

        private void ApplySettings()
        {
            try
            {
                this.Cursor = Cursors.WaitCursor;

                // Apply general settings.
                this.imageNavigator.ForeColor = this.settings.General.ForeColor;
                this.imageNavigator.BackColor = this.settings.General.BackColor;
                this.imageNavigator.Font = this.settings.General.Font;
                this.imageDisplay.ForeColor = this.settings.General.ForeColor;
                this.imageDisplay.BackColor = this.settings.General.BackColor;
                this.imageDisplay.Font = this.settings.General.Font;

                // Apply zooming settings.
                this.imageDisplay.ErrorText = this.settings.Zooming.ErrorText;
                this.imageDisplay.Interpolation = this.settings.Zooming.Interpolation;
                this.imageDisplay.MinimumZoomFactor = this.settings.Zooming.MinimumFactor;
                this.imageDisplay.MaximumZoomFactor = this.settings.Zooming.MaximumFactor;
                this.imageDisplay.DefaultZoomFactor = this.settings.Zooming.DefaultFactor;
                this.imageDisplay.Padding = new Padding(this.settings.Zooming.ImagePadding);

                // Apply filtering settings.
                this.imageNavigator.UpdateFilterSettings();

                // Reset zoom factor combobox.
                this.UpdateZoomFactors();

                // Apply image dependent settings.
                this.UpdateDisplay();
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.Cursor = Cursors.Default;
            }
        }

        private bool IsValidPath(string argument)
        {
            try
            {
                if (!String.IsNullOrEmpty(argument))
                {
                    // Firstly, assume that given argument is a file.
                    FileInfo file = new FileInfo(argument);
                    if (!file.Exists)
                    {
                        // Secondly, assume that given argument is a folder.
                        DirectoryInfo folder = new DirectoryInfo(argument);
                        return folder.Exists;
                    }
                    else
                    {
                        return true;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private void LoadNotes(FileNode fileNode)
        {
            try
            {
                string filename = null;
                if (fileNode != null)
                {
                    FileInfo fileInfo = fileNode.NodeInfo<FileInfo>();
                    if (fileInfo != null)
                    {
                        filename = fileInfo.FullName;
                    }
                }

                this.notesPanel.Enabled = (fileNode != null);

                this.notesPanel.LoadNotes(filename);
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        #endregion // Private member implementation.
    }
}
