﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Collections.Generic;

namespace plexdata.ImageViewer
{
    partial class MainForm
    {
        /// <summary>
        /// Helper class that represents one 
        /// single zoom factor combobox item.
        /// </summary>
        private class ZoomFactorItem
        {
            public ZoomFactorItem(int factor)
                : this(String.Format("{0}%", factor), factor)
            {
            }

            public ZoomFactorItem(string label, int factor)
                : base()
            {
                this.Label = label;
                this.Factor = factor;
            }

            public string Label { get; private set; }

            public int Factor { get; private set; }

            public override int GetHashCode()
            {
                return base.GetHashCode();
            }

            public override bool Equals(object other)
            {
                if (other is ZoomFactorItem)
                {
                    return this.Factor == (other as ZoomFactorItem).Factor;
                }
                else if (other is int)
                {
                    return this.Factor == (float)((int)other);
                }
                else
                {
                    return base.Equals(other);
                }
            }

            public override string ToString()
            {
                if (String.IsNullOrEmpty(this.Label))
                {
                    return this.Factor.ToString();
                }
                else
                {
                    return this.Label;
                }
            }

            public int ToInteger()
            {
                return (int)this.Factor;
            }

        }

        /// <summary>
        /// The instance of the list containing all predefined 
        /// zoom factor values.
        /// </summary>
        private List<ZoomFactorItem> zoomFactorList = null;

        /// <summary>
        /// Indicates if the navigator tree must be restored 
        /// after maximizing/minimizing the main window.
        /// </summary>
        private bool restoreNavigator = false;

        #region Private helper function implementations.

        private void InitializeToolbars()
        {
            this.mainContainer.TopToolStripPanel.Controls.Clear();
            this.mainContainer.TopToolStripPanel.Join(this.zoomToolbar, new Point(0, 0));
            this.mainContainer.TopToolStripPanel.Join(this.mainToolbar, new Point(0, 0));

            // Initialize zoom factor combobox.
            this.UpdateZoomFactors();

            // Assign image display event handlers.
            this.imageDisplay.ImageChanged += this.TBImageDisplayImageChanged;
            this.imageDisplay.ZoomFactorChanged += this.TBImageDisplayZoomFactorChanged;

            // Assign main toolbar event handlers.
            this.tbbExit.Click += this.TBExitButtonClick;
            this.tbbNavigator.Click += this.TBNavigatorButtonClick;
            this.tbbFullscreen.Click += this.TBFullscreenButtonClick;
            this.tbbClipboard.Click += this.TBClipboardButtonClick;
            this.tbbSlideshow.Click += this.TBSlideshowButtonClick;
            this.tbbSettings.Click += this.TBSettingsButtonClick;
            this.tbbHelp.Click += this.TBHelpButtonClick;
            this.tbmHelp.Click += this.TBHelpMenuClick;
            this.tbmAbout.Click += this.TBAboutMenuClick;

            // Assign zoom toolbar event handlers.
            this.tbbZoomIn.Click += this.TBZoomInButtonClick;
            this.tbbZoomOut.Click += this.TBZoomOutButtonClick;
            this.tbbFitSize.Click += this.TBFitSizeButtonClick;
            this.tbbFitWidth.Click += this.TBFitWidthButtonClick;
            this.tbbRotateLeft.Click += this.TBRotateLeftButtonClick;
            this.tbbRotateRight.Click += this.TBRotateRightButtonClick;
            this.tbbJumpPrevious.Click += this.TBJumpPreviousButtonClick;
            this.tbbJumpNext.Click += this.TBJumpNextButtonClick;
            this.tbbNotes.Click += this.TBNotesButtonClick;

            // Assign zoom factor combobox event handlers.
            this.tbcZoomFactors.Leave += this.TBZoomFactorsComboLeave;
            this.tbcZoomFactors.KeyPress += this.TBZoomFactorsComboKeyPress;
            this.tbcZoomFactors.SelectedIndexChanged += this.TBZoomFactorsComboIndexChanged;

            // Adjust current toolbar states.
            this.UpdateMainToolbarState();
            this.UpdateZoomToolbarState();
        }

        private void UpdateZoomFactors()
        {
            // Save last chosen combobox item for a later use.
            ZoomFactorItem current = (this.tbcZoomFactors.SelectedItem as ZoomFactorItem);

            // Recreate zoom factor list.
            this.zoomFactorList = new List<ZoomFactorItem>();

            int standard = this.imageDisplay.DefaultZoomFactor;
            int minimum = this.imageDisplay.MinimumZoomFactor;
            int maximum = this.imageDisplay.MaximumZoomFactor;

            for (int factor = minimum; factor < standard; factor *= 2)
            {
                this.zoomFactorList.Add(new ZoomFactorItem(factor));
            }

            // Ensure the availability of default zoom factor 
            // even if it is not included in the calculation!
            for (int factor = standard; factor < maximum; factor *= 2)
            {
                this.zoomFactorList.Add(new ZoomFactorItem(factor));
            }

            this.zoomFactorList.Add(new ZoomFactorItem(maximum));

            // Initialize zoom factor combobox.
            this.tbcZoomFactors.Items.Clear();
            this.tbcZoomFactors.Items.AddRange(this.zoomFactorList.ToArray());

            // Reuse last chosen combobox item if possible.
            if (current != null && this.tbcZoomFactors.Items.Contains(current))
            {
                this.tbcZoomFactors.SelectedItem = current;
            }
            else
            {
                this.tbcZoomFactors.SelectedItem = standard;
            }
        }

        private void UpdateMainToolbarState()
        {
            bool enabled = (this.imageDisplay.Image != null);
            this.tbbSlideshow.Enabled = enabled;
        }

        private void UpdateMainToolbarClipboardButtonState()
        {
            try
            {
                // BUG: Clipboard observer also recognizes files that are copied by e.g. the Windows Explorer. 
                // At the moment there is no (proper) way to solve this issue.
                bool enabled = false;
                if (Clipboard.ContainsFileDropList())
                {
                    List<string> pattern = Program.MainForm.Settings.Filtering.GetSearchPattern(true);
                    foreach (string item in Clipboard.GetFileDropList())
                    {
                        if (Program.InSearchPattern(Path.GetExtension(item), pattern))
                        {
                            enabled = true;
                            break;
                        }
                    }
                }

                this.tbbClipboard.Enabled = enabled;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void UpdateZoomToolbarState()
        {
            bool enabled = (this.imageDisplay.Image != null);
            this.tbbZoomIn.Enabled = enabled;
            this.tbbZoomOut.Enabled = enabled;
            this.tbcZoomFactors.Enabled = enabled;
            this.tbbFitSize.Enabled = enabled;
            this.tbbFitWidth.Enabled = enabled;
            this.tbbRotateLeft.Enabled = enabled;
            this.tbbRotateRight.Enabled = enabled;
        }

        private void ChooseNewZoomFactor()
        {
            string current = this.tbcZoomFactors.Text;
            if (!String.IsNullOrEmpty(current))
            {
                current = current.Replace(" ", "").Replace("%", "").ToLower();
                if (!String.IsNullOrEmpty(current))
                {
                    foreach (ZoomFactorItem item in this.zoomFactorList)
                    {
                        if (item.Label.ToLower() == current || item.Label.ToLower() == current + "%")
                        {
                            this.imageDisplay.ZoomFactor = item.Factor;
                            return; // Done!
                        }
                    }

                    int factor = 0;
                    if (int.TryParse(current, out factor))
                    {
                        if (this.imageDisplay.IsValidZoomFactor(factor))
                        {
                            this.imageDisplay.ZoomFactor = factor;
                            return; // Done!
                        }
                    }

                    // Use default zoom factor in any other case.
                    this.imageDisplay.ZoomFactor = this.imageDisplay.DefaultZoomFactor;
                    this.tbcZoomFactors.Text = this.imageDisplay.ZoomFactor.ToString() + "%";
                }
            }
        }

        #endregion // Private helper function implementations.

        #region Image display event handler implementation.

        private void TBImageDisplayImageChanged(object sender, EventArgs args)
        {
            this.UpdateMainToolbarState();
            this.UpdateZoomToolbarState();
        }

        private void TBImageDisplayZoomFactorChanged(object sender, EventArgs args)
        {
            this.tbcZoomFactors.Text = this.imageDisplay.ZoomFactor.ToString() + "%";
        }

        #endregion // Image display event handler implementation.

        #region Main toolbar event handler implementation.

        private void TBExitButtonClick(object sender, EventArgs args)
        {
            this.Close();
        }

        private void TBNavigatorButtonClick(object sender, EventArgs args)
        {
            this.mainSplitter.Panel1Collapsed = !this.mainSplitter.Panel1Collapsed;
            if (this.mainSplitter.Panel1Collapsed)
            {
                this.tbbNavigator.Image = global::plexdata.ImageViewer.Properties.Resources.ShowNavigateIcon;
                this.imageDisplay.Select();
            }
            else
            {
                this.tbbNavigator.Image = global::plexdata.ImageViewer.Properties.Resources.HideNavigateIcon;
            }
        }

        private void TBFullscreenButtonClick(object sender, EventArgs args)
        {
            if (this.maximizer.IsMaximized)
            {
                this.maximizer.Restore();

                if (this.restoreNavigator)
                {
                    this.restoreNavigator = false;
                    this.tbbNavigator.PerformClick();
                }
            }
            else
            {
                this.maximizer.Maximize();

                if (!this.mainSplitter.Panel1Collapsed)
                {
                    this.restoreNavigator = true;
                    this.tbbNavigator.PerformClick();
                }
            }
        }

        private void TBClipboardButtonClick(object sender, EventArgs args)
        {
            CopyListDialog dialog = new CopyListDialog();
            dialog.ShowDialog(this);
        }

        private void TBSlideshowButtonClick(object sender, EventArgs args)
        {
            this.StartSlideshow();
        }

        private void TBSettingsButtonClick(object sender, EventArgs args)
        {
            SettingsDialog dialog = new SettingsDialog(this.settings);
            if (DialogResult.OK == dialog.ShowDialog(this))
            {
                this.settings = new Settings(dialog.Settings);
                this.ApplySettings();
            }
        }

        private void TBHelpButtonClick(object sender, EventArgs args)
        {
            this.tbbHelp.ShowDropDown();
        }

        private void TBHelpMenuClick(object sender, EventArgs args)
        {
            Program.ShowHelp();
        }

        private void TBAboutMenuClick(object sender, EventArgs args)
        {
            AboutBox about = new AboutBox();
            about.ShowDialog();
        }

        #endregion // Main toolbar event handler implementation.

        #region Zoom toolbar event handler implementation.

        private void TBZoomInButtonClick(object sender, EventArgs args)
        {
            this.imageDisplay.ZoomIn();
        }

        private void TBZoomOutButtonClick(object sender, EventArgs args)
        {
            this.imageDisplay.ZoomOut();
        }

        private void TBFitSizeButtonClick(object sender, EventArgs args)
        {
            this.imageDisplay.FitToSize();
        }

        private void TBFitWidthButtonClick(object sender, EventArgs args)
        {
            this.imageDisplay.FitToWidth();
        }

        private void TBRotateLeftButtonClick(object sender, EventArgs args)
        {
            this.imageDisplay.RotateLeft();
        }

        private void TBRotateRightButtonClick(object sender, EventArgs args)
        {
            this.imageDisplay.RotateRight();
        }

        private void TBJumpPreviousButtonClick(object sender, EventArgs args)
        {
            this.imageNavigator.JumpPrevious();
        }

        private void TBJumpNextButtonClick(object sender, EventArgs args)
        {
            this.imageNavigator.JumpNext();
        }

        private void TBNotesButtonClick(object sender, EventArgs args)
        {
            this.dispSplitter.Panel2Collapsed = !this.dispSplitter.Panel2Collapsed;

            // Always load external notes file to ensure an auto-save of currently made changes.
            this.LoadNotes(this.imageNavigator.SelectedNode as FileNode);
        }

        #endregion // Zoom toolbar event handler implementation.

        #region Zoom factor combobox event handler implementation.

        private void TBZoomFactorsComboLeave(object sender, EventArgs args)
        {
            this.ChooseNewZoomFactor();
        }

        private void TBZoomFactorsComboKeyPress(object sender, KeyPressEventArgs args)
        {
            if (args.KeyChar == (char)Keys.Return)
            {
                this.ChooseNewZoomFactor();
            }
        }

        private void TBZoomFactorsComboIndexChanged(object sender, EventArgs args)
        {
            ZoomFactorItem item = (this.tbcZoomFactors.SelectedItem as ZoomFactorItem);
            if (item != null)
            {
                this.imageDisplay.ZoomFactor = item.Factor;
            }
        }

        #endregion // Zoom factor combobox event handler implementation.

        #region Toolbar maintenance event handlers.

        private void TBZoomToolbarLocationChanged(object sender, EventArgs args)
        {
            this.settings.Maintain.ZoomToolbar = this.zoomToolbar.Location;
        }

        private void TBMainToolbarLocationChanged(object sender, EventArgs args)
        {
            this.settings.Maintain.MainToolbar = this.mainToolbar.Location;
        }

        #endregion // Toolbar maintenance event handlers.
    }
}
