﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Threading;
using System.Diagnostics;
using System.Windows.Forms;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;

namespace plexdata.ImageViewer
{
    partial class MainForm
    {
        private Image tempImage = null;

        private void InitializeDisplay()
        {
            this.imageNavigator.AfterSelect += this.DisplayNavigatorAfterSelect;

            this.imageDisplay.VerticalScroll.SmallChange = 20;
            this.imageDisplay.HorizontalScroll.SmallChange = 20;

            // Pre-initialize the image display's context menu.
            this.imageDisplay.ContextMenu = new ContextMenu();
            this.imageDisplay.ContextMenu.Popup += new EventHandler(this.ImageDisplayContextMenuPopup);
        }

        private void UpdateDisplay()
        {
            if (this.settings.General.AutoAdjust && this.imageDisplay.Image != null &&
               (this.imageDisplay.Image.Size.Width > this.imageDisplay.ClientSize.Width ||
                this.imageDisplay.Image.Size.Height > this.imageDisplay.ClientSize.Height))
            {
                this.imageDisplay.FitToSize();
            }
        }

        private void DisplayNavigatorAfterSelect(object sender, TreeViewEventArgs args)
        {
            Cursor oldCursor = Program.MainForm.Cursor;
            Program.MainForm.Cursor = Cursors.WaitCursor;
            try
            {
                // Redraw entire main window!
                this.Invalidate();
                this.Update();

                // Create delay load handler thread.
                Thread delayHandler = new Thread(this.AsyncLoadImage);
                delayHandler.IsBackground = true;
                delayHandler.Start(args.Node as FileNode);

                // Run delay load behavior.
                Thread.Sleep(100);
                if (delayHandler.IsAlive)
                {
                    // Set loading mode!
                    this.imageDisplay.Loading = true;
                    while (delayHandler.IsAlive)
                    {
                        Thread.Sleep(10);
                    }
                }

                // Set newly loaded image.
                this.imageDisplay.SuspendLayout();
                this.imageDisplay.Image = this.tempImage;

                // Apply specific settings if needed.
                this.UpdateDisplay();

                // Additionally, try loading belonging file notes.
                this.LoadNotes(args.Node as FileNode);
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                this.imageDisplay.Image = null;
            }
            finally
            {
                // Reset loading mode!
                this.imageDisplay.Loading = false;
                this.imageDisplay.ResumeLayout();
                this.imageDisplay.PerformLayout();
                Program.MainForm.Cursor = oldCursor;
            }
        }

        private void AsyncLoadImage(object parameter)
        {
            try
            {
                // Release old temporary image.
                if (this.tempImage != null)
                {
                    this.tempImage.Dispose();
                    this.tempImage = null;
                }

                if (parameter != null && parameter is FileNode)
                {
                    FileInfo fileInfo = (parameter as FileNode).NodeInfo<FileInfo>();
                    if (fileInfo != null)
                    {
                        // Load new temporary image.
                        using (FileStream fileStream = fileInfo.OpenRead())
                        {
                            if (Program.MainForm.Settings.General.AutoRotate)
                            {
                                this.tempImage = RotateImageHelper.RotateFlip(Image.FromStream(fileStream));
                            }
                            else
                            {
                                this.tempImage = Image.FromStream(fileStream);
                            }
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                this.tempImage = null;
            }
        }

        #region Image display context menu dependent member implementation.

        private void ImageDisplayContextMenuPopup(object sender, EventArgs args)
        {
            this.imageDisplay.ContextMenu.MenuItems.Clear();

            if (this.imageDisplay.Image != null)
            {
                MenuItem item;

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_DEFAULT_SIZE, this.DefaultZoomMenuItemClick);
                item.DefaultItem = true;
                this.imageDisplay.ContextMenu.MenuItems.Add(item);

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_ZOOM_IN, this.ZoomInMenuItemClick);
                item.Enabled = (this.imageDisplay.ZoomFactor < this.imageDisplay.MaximumZoomFactor);
                this.imageDisplay.ContextMenu.MenuItems.Add(item);

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_ZOOM_OUT, this.ZoomOutMenuItemClick);
                item.Enabled = (this.imageDisplay.ZoomFactor > this.imageDisplay.MinimumZoomFactor);
                this.imageDisplay.ContextMenu.MenuItems.Add(item);

                this.imageDisplay.ContextMenu.MenuItems.Add(new MenuItem("-"));

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_FIT_WINDOW, this.PageSizeMenuItemClick);
                this.imageDisplay.ContextMenu.MenuItems.Add(item);

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_FIT_WIDTH, this.PageWidthMenuItemClick);
                this.imageDisplay.ContextMenu.MenuItems.Add(item);

                this.imageDisplay.ContextMenu.MenuItems.Add(new MenuItem("-"));

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_SMOOTH, this.InterpolationMenuItemClick);
                item.Checked = this.settings.Zooming.Interpolation;
                this.imageDisplay.ContextMenu.MenuItems.Add(item);

                this.imageDisplay.ContextMenu.MenuItems.Add(new MenuItem("-"));

                item = new MenuItem(Properties.Resources.IMAGEDISPLAY_MENUITEM_PROPERTIES, this.PropertiesMenuItemClick);
                this.imageDisplay.ContextMenu.MenuItems.Add(item);
            }
        }

        private void DefaultZoomMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                this.imageDisplay.ZoomFactor = this.imageDisplay.DefaultZoomFactor;
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void ZoomInMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                this.imageDisplay.ZoomIn();
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void ZoomOutMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                this.imageDisplay.ZoomOut();
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void PageSizeMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                this.imageDisplay.FitToSize();
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void PageWidthMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                this.imageDisplay.FitToWidth();
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void InterpolationMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                MenuItem item = (sender as MenuItem);
                if (item != null)
                {
                    this.imageDisplay.Interpolation = !item.Checked;
                    this.settings.Zooming.Interpolation = !item.Checked;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void PropertiesMenuItemClick(object sender, EventArgs args)
        {
            try
            {
                FileNode fileNode = (this.imageNavigator.SelectedNode as FileNode);
                if (fileNode != null)
                {
                    FileInfo fileInfo = fileNode.NodeInfo<FileInfo>();
                    if (fileInfo != null)
                    {
                        ShellAccess.ShowPropertiesDialog(fileInfo.FullName);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        #endregion // Image display context menu dependent member implementation.
    }
}
