﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Linq;
using System.Drawing;
using System.Reflection;
using System.Collections;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;
using System.Collections.Generic;
using Microsoft.VisualBasic.FileIO;
using System.Collections.Specialized;

using plexdata.Utilities;

namespace plexdata.ImageViewer
{
    public partial class ImageNavigator : TreeView
    {
        // General remarks and behavior description!
        // Tree nodes are not added completely at first initialization. This 
        // means in fact that only the root nodes are added. Furthermore, to 
        // ensure file system observation all available drive sub-nodes are 
        // added to the MyComputer root node.
        // 
        // If then one of the shown root nodes is expanded the corresponding 
        // child nodes are added! Collapsing a node with existing children 
        // will not remove those sub-nodes.
        // 
        // Another way of adding child nodes comes through the running file 
        // system observers. In this conjunction it will be possible that 
        // child nodes are changed, appear or also disappear. Appearing of 
        // child nodes happens for example when a file, which is currently 
        // filtered out by the active search patterns, is renamed using the 
        // Windows Explore and this file gets suddenly an extension witch fit 
        // the active search patterns. Disappearing on the other hand may 
        // happen for instance if a currently shown file is deleted in the 
        // Windows Explorer or if it is renamed using an extension that does 
        // not longer fit active search patterns.

        private List<FileSystemWatcher> observers = new List<FileSystemWatcher>();

        private NavigatorNodeCollection collection = null;
        private NavigatorNode currentNode = null;

        private List<string> searchPattern = new List<string>();
        private List<string> excludedPattern = new List<string>();

        #region Context menu related member and constant declaration.

        private ContextMenu menuFavorites = null;
        private ContextMenu menuFolder = null;
        private ContextMenu menuFile = null;

        private MenuItem menuItemRemoveFavorite = null;
        private MenuItem menuItemClearAllFavorites = null;
        private MenuItem menuItemAddToFavorites = null;
        private MenuItem menuItemPathOpen = null;
        private MenuItem menuItemPathCopy = null;
        private MenuItem menuItemFileShow = null;
        private MenuItem menuItemFileEdit = null;
        private MenuItem menuItemProperties = null;
        private MenuItem menuItemExpand = null; // Never use an "Expand All" menu item!
        private MenuItem menuItemCollapse = null;
        private MenuItem menuItemLocate = null;
        private MenuItem menuItemDelete = null;
        private MenuItem menuItemCopyList = null;
        private MenuItem menuItemCopyListInsert = null;
        private MenuItem menuItemCopyListRemove = null;
        private MenuItem menuItemCopyListEmpty = null;
        private MenuItem menuItemCopyListContent = null;

        #endregion // Context menu related member and constant declaration.

        #region Image list related constant declarations.

        public const string NODEICON_DEFAULT = "NodeIconDefault";
        public const string ROOTICON_FAVORITES = "RootIconFavorites";
        public const string ROOTICON_DESKTOP = "RootIconDesktop";
        public const string ROOTICON_PICTURES = "RootIconPictures";
        public const string ROOTICON_COMPUTER = "RootIconComputer";
        public const string DRIVEICON_FIXED = "DriveIconFixed";
        public const string DRIVEICON_COMPACT = "DriveIconCompact";
        public const string DRIVEICON_NETWORK = "DriveIconNetwork";
        public const string DRIVEICON_REMOVABLE = "DriveIconRemovable";
        public const string FOLDERICON_FAVORITE = "FolderIconFavorite";
        public const string FOLDERICON_STANDARD = "FolderIconStandard";
        public const string FILEICON_PICTURE = "FileIconPicture";

        #endregion // Image list related constant declarations.

        // BUG: Icons of expanded nodes are not drawn properly when scrolling!
        // This is because of using images with size 16x16 pixels but the 
        // item height is set to 20 pixels!

        public ImageNavigator()
            : base()
        {
            this.InitializeComponent();

            // Enable Double Buffering to reduce flickering.
            PropertyInfo property = typeof(TreeView).GetProperty("DoubleBuffered", BindingFlags.NonPublic | BindingFlags.Instance);
            property.SetValue(this, true, null);
        }

        private static ComponentResourceManager resourceManager = null;
        public static ComponentResourceManager ResourceManager
        {
            get
            {
                if (ImageNavigator.resourceManager == null)
                {
                    ImageNavigator.resourceManager = new ComponentResourceManager(typeof(ImageNavigator));
                }
                return ImageNavigator.resourceManager;
            }
        }

        protected override void Dispose(bool disposing)
        {
            if (disposing)
            {
                // Release all observers!
                foreach (FileSystemWatcher watcher in this.observers)
                {
                    try
                    {
                        watcher.EnableRaisingEvents = false;
                        watcher.Dispose();
                    }
                    catch (Exception exception)
                    {
                        Debug.WriteLine(exception);
                    }
                }
                this.observers.Clear();
            }

            base.Dispose(disposing);
        }

        #region Public member implementation.

        public void InitializeRootNodes()
        {
            NavigatorNode node;

            this.Nodes.Clear();

            node = new FavoritesRoot();
            node.ContextMenu = this.menuFavorites;
            this.Nodes.Add(node);

            node = new DesktopRoot();
            node.ContextMenu = this.menuFolder;
            this.Nodes.Add(node);

            node = new PrivatePicturesRoot();
            node.ContextMenu = this.menuFolder;
            this.Nodes.Add(node);

            node = new CommonPicturesRoot();
            node.ContextMenu = this.menuFolder;
            this.Nodes.Add(node);

            node = new ComputerRoot();
            node.ContextMenu = this.menuFolder;
            this.Nodes.Add(node);

            // Don't forget to update current item height!
            this.SetupItemHeight();

            // Needed since supporting File System Watchers.
            this.AddDriveNodes(node);
        }

        public void UpdateFilterSettings()
        {
            try
            {
                this.BeginUpdate();

                this.searchPattern = this.Filtering.GetSearchPattern(true);
                this.excludedPattern = this.Filtering.GetSearchPattern(false);

                // Add first and remove afterwards!
                this.UpdateEnabledItems();
                this.RemoveDisabledItems();

                this.ResetObservers();
            }
            finally
            {
                this.EndUpdate();
            }
        }

        public void ExpandToPath(string fullname)
        {
            try
            {
                if (!String.IsNullOrEmpty(fullname))
                {
                    foreach (NavigatorNode root in this.Nodes)
                    {
                        if (root.IsComputer)
                        {
                            this.AutoExpand(root, fullname);

                            // Choose the first file if given "fullname" is a folder.
                            if (this.SelectedNode != null)
                            {
                                if (this.SelectedNode.IsFolderNode)
                                {
                                    this.SelectedNode.Expand();
                                    if (this.SelectedNode.HasChildren && this.SelectedNode.FirstNode.IsFileNode)
                                    {
                                        this.SelectedNode = this.SelectedNode.FirstNode;
                                    }
                                }
                            }
                            return; // Done!
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        public void JumpNext()
        {
            this.JumpNext(this.SelectedNode);
        }

        public void JumpPrevious()
        {
            this.JumpPrevious(this.SelectedNode);
        }

        #endregion // Public member implementation.

        #region Public properties implementation.

        public new NavigatorNodeCollection Nodes
        {
            get
            {
                if (this.collection == null)
                {
                    this.collection = new NavigatorNodeCollection(base.Nodes);
                }
                return this.collection;
            }
        }

        public new NavigatorNode SelectedNode
        {
            get { return (base.SelectedNode as NavigatorNode); }
            set { base.SelectedNode = value; }
        }

        #endregion // Public properties implementation.

        #region Overwritten event handler implementation.

        protected override void OnFontChanged(EventArgs args)
        {
            base.OnFontChanged(args);
            this.SetupItemHeight();
        }

        protected override void OnBeforeExpand(TreeViewCancelEventArgs args)
        {
            base.OnBeforeExpand(args);

            Cursor oldCursor = Program.MainForm.Cursor;
            Program.MainForm.Cursor = Cursors.WaitCursor;
            try
            {
                this.BeginUpdate();

                NavigatorNode node = (args.Node as NavigatorNode);
                if (node != null)
                {
                    if (node.FirstNode.IsDefaultNode)
                    {
                        node.FirstNode.Remove();
                        this.AddChildNodes(node);
                    }
                }
            }
            finally
            {
                Program.MainForm.Cursor = oldCursor;
                this.EndUpdate();
            }
        }

        protected override void OnNodeMouseClick(TreeNodeMouseClickEventArgs args)
        {
            base.OnNodeMouseClick(args);

            if (args.Button == MouseButtons.Right)
            {
                // Save last right clicked tree node!
                this.currentNode = (args.Node as NavigatorNode);
            }
            else
            {
                this.currentNode = null;
            }
        }

        protected override void OnPreviewKeyDown(PreviewKeyDownEventArgs args)
        {
            base.OnPreviewKeyDown(args);

            if (args.KeyCode == Keys.Apps)
            {
                // Save last right clicked tree node!
                this.currentNode = this.SelectedNode;
            }
            else
            {
                this.currentNode = null;
            }
        }

        #endregion // Overwritten event handler implementation.

        #region General node handling implementation.

        private void AddChildNodes(NavigatorNode parent)
        {
            try
            {
                if (parent != null)
                {
                    if (parent.IsFavorites)
                    {
                        this.AddFavoriteNodes(parent);
                    }
                    else if (parent.IsDesktop)
                    {
                        this.AddFolderNodes(parent);
                    }
                    else if (parent.IsPictures)
                    {
                        this.AddFolderNodes(parent);
                    }
                    else if (parent.IsDriveNode)
                    {
                        this.AddFolderNodes(parent);
                    }
                    else if (parent.IsFolderNode)
                    {
                        this.AddFolderNodes(parent);
                    }
                    else if (parent.IsFileNode)
                    {
                        Debug.Assert(false);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void AddFavoriteNodes(NavigatorNode parent)
        {
            try
            {
                if (parent != null)
                {
                    foreach (Favorite favorite in this.Maintain.Favorites)
                    {
                        FavoriteNode node = new FavoriteNode(favorite);
                        node.ContextMenu = this.menuFavorites;
                        parent.Nodes.Add(node);
                    }
                    parent.Nodes.Sort();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void AddDriveNodes(NavigatorNode parent)
        {
            try
            {
                if (parent != null)
                {
                    // Remove default node if still exists.
                    if (parent.FirstNode.IsDefaultNode)
                    {
                        parent.FirstNode.Remove();
                    }

                    // Now add all currently available drives.
                    foreach (string drive in Environment.GetLogicalDrives())
                    {
                        // REMARK: Don't sort the drive nodes.
                        DriveNode node = new DriveNode(drive);
                        if (node.IsDriveReady)
                        {
                            node.ContextMenu = this.menuFolder;
                            parent.Nodes.Add(node);
                        }
                    }

                    // Reset currently running file system observers.
                    this.ResetObservers();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void AddFolderNodes(NavigatorNode parent)
        {
            try
            {
                if (parent != null)
                {
                    DirectoryInfo info = parent.NodeInfo<DirectoryInfo>();
                    if (info != null && info.Exists)
                    {
                        foreach (DirectoryInfo folder in info.GetDirectories())
                        {
                            if (this.CanAddFolder(folder))
                            {
                                FolderNode node = new FolderNode(folder);
                                node.ContextMenu = this.menuFolder;
                                parent.Nodes.Add(node);
                            }
                        }
                        this.AddFileNodes(parent);

                        parent.Nodes.Sort();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void AddFileNodes(NavigatorNode parent)
        {
            try
            {
                if (parent != null)
                {
                    DirectoryInfo info = parent.NodeInfo<DirectoryInfo>();
                    if (info != null && info.Exists)
                    {
                        // REMARK: Sorting is done in member function AddFolderNodes()!
                        NavigatorNode[] nodes = this.CollectFileNodes(info);
                        if (nodes.Length > 0)
                        {
                            parent.Nodes.AddRange(nodes);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private NavigatorNode[] CollectFileNodes(DirectoryInfo info)
        {
            return this.CollectFileNodes(info, this.searchPattern);
        }

        private NavigatorNode[] CollectFileNodes(DirectoryInfo info, List<string> pattern)
        {
            List<NavigatorNode> result = new List<NavigatorNode>();
            try
            {
                if (info != null && info.Exists)
                {
                    foreach (FileInfo file in info.GetFiles())
                    {
                        if (Program.InSearchPattern(file.Extension, pattern))
                        {
                            FileNode node = new FileNode(file);
                            node.ContextMenu = this.menuFile;
                            result.Add(node);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                result.Clear();
            }
            return result.ToArray();
        }

        private NavigatorNode[] AffectedFolderNodes(string fullname)
        {
            List<NavigatorNode> result = new List<NavigatorNode>();
            try
            {
                DirectoryInfo candidate = new DirectoryInfo(fullname);
                if (candidate != null)
                {
                    foreach (NavigatorNode current in this.Nodes.Find(NavigatorNode.FOLDER_NODE, true))
                    {
                        if (current.CompareTo(candidate) == 0)
                        {
                            result.Add(current);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                result.Clear();
            }
            return result.ToArray();
        }

        private NavigatorNode[] AffectedFileNodes(string fullname)
        {
            List<NavigatorNode> result = new List<NavigatorNode>();
            try
            {
                FileInfo candidate = new FileInfo(fullname);
                if (candidate != null)
                {
                    foreach (NavigatorNode current in this.Nodes.Find(candidate.Extension, true))
                    {
                        if (current.CompareTo(candidate) == 0)
                        {
                            result.Add(current);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                result.Clear();
            }
            return result.ToArray();
        }

        #endregion // General node handling implementation.

        #region Other private member implementation.

        private void InitializeComponent()
        {
            #region Prepare navigator tree icons.

            if (this.ImageList != null)
            {
                this.ImageList.Dispose();
                this.ImageList = null;
            }

            this.ImageList = new ImageList();

            this.ImageList.Images.Add(
                ImageNavigator.NODEICON_DEFAULT,
                global::plexdata.ImageViewer.Properties.Resources.NodeIconDefault);
            this.ImageList.Images.Add(
                ImageNavigator.ROOTICON_FAVORITES,
                global::plexdata.ImageViewer.Properties.Resources.RootIconFavorites);
            this.ImageList.Images.Add(
                ImageNavigator.ROOTICON_DESKTOP,
                global::plexdata.ImageViewer.Properties.Resources.RootIconDesktop);
            this.ImageList.Images.Add(
                ImageNavigator.ROOTICON_PICTURES,
                global::plexdata.ImageViewer.Properties.Resources.RootIconPictures);
            this.ImageList.Images.Add(
                ImageNavigator.ROOTICON_COMPUTER,
                global::plexdata.ImageViewer.Properties.Resources.RootIconComputer);
            this.ImageList.Images.Add(
                ImageNavigator.DRIVEICON_FIXED,
                global::plexdata.ImageViewer.Properties.Resources.DriveIconFixed);
            this.ImageList.Images.Add(
                ImageNavigator.DRIVEICON_COMPACT,
                global::plexdata.ImageViewer.Properties.Resources.DriveIconCompact);
            this.ImageList.Images.Add(
                ImageNavigator.DRIVEICON_NETWORK,
                global::plexdata.ImageViewer.Properties.Resources.DriveIconNetwork);
            this.ImageList.Images.Add(
                ImageNavigator.DRIVEICON_REMOVABLE,
                global::plexdata.ImageViewer.Properties.Resources.DriveIconRemovable);
            this.ImageList.Images.Add(
                ImageNavigator.FOLDERICON_FAVORITE,
                global::plexdata.ImageViewer.Properties.Resources.FolderIconFavorite);
            this.ImageList.Images.Add(
                ImageNavigator.FOLDERICON_STANDARD,
                global::plexdata.ImageViewer.Properties.Resources.FolderIconStandard);
            this.ImageList.Images.Add(
                ImageNavigator.FILEICON_PICTURE,
                global::plexdata.ImageViewer.Properties.Resources.FileIconPicture);

            #endregion // Prepare navigator tree icons.

            #region Prepare all context menu items.

            this.menuItemRemoveFavorite = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_REMOVEFAVORITE"),
                this.OnMenuItemRemoveFavorite);

            this.menuItemClearAllFavorites = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_CLEARALLFAVORITES"),
                this.OnMenuItemClearAllFavorites);

            this.menuItemAddToFavorites = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_ADDTOFAVORITES"),
                this.OnMenuItemAddToFavorites);

            this.menuItemPathOpen = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_PATHOPEN"),
                this.OnMenuItemPathOpen);

            this.menuItemPathCopy = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_PATHCOPY"),
                this.OnMenuItemPathCopy);

            this.menuItemFileShow = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_FILESHOW"),
                this.OnMenuItemFileShow);

            this.menuItemFileEdit = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_FILEEDIT"),
                this.OnMenuItemFileEdit);

            this.menuItemProperties = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_PROPERTIES"),
                this.OnMenuItemProperties);

            this.menuItemExpand = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_EXPAND"),
                this.OnMenuItemExpand);

            this.menuItemCollapse = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_COLLAPSE"),
                this.OnMenuItemCollapse);

            this.menuItemLocate = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_LOCATE"),
                this.OnMenuItemLocate);

            this.menuItemDelete = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_DELETE"),
                this.OnMenuItemDelete);

            this.menuItemCopyList = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_COPYLIST"));

            this.menuItemCopyListInsert = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_COPYLIST_INSERT"),
                this.OnMenuItemCopyListInsert);

            this.menuItemCopyListRemove = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_COPYLIST_REMOVE"),
                this.OnMenuItemCopyListRemove);

            this.menuItemCopyListEmpty = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_COPYLIST_EMPTY"),
                this.OnMenuItemCopyListEmpty);

            this.menuItemCopyListContent = new MenuItem(
                ResourceManager.GetString("MENUITEMTEXT_COPYLIST_CONTENT"),
                this.OnMenuItemCopyListContent);

            #endregion // Prepare all context menu items.

            #region Prepare all context menus.

            // REMARK: Do not add any menu item here!
            // This is because that different context menus cannot share 
            // one and the same menu item. A copy must be used instead 
            // using member function CloneMenu(). See also the notes in 
            // the help for class ContextMenu.

            this.menuFavorites = new ContextMenu();
            this.menuFavorites.Popup += this.ContextMenuPopupHandler;

            this.menuFolder = new ContextMenu();
            this.menuFolder.Popup += this.ContextMenuPopupHandler;

            this.menuFile = new ContextMenu();
            this.menuFile.Popup += this.ContextMenuPopupHandler;

            #endregion // Prepare all context menus.
        }

        private void UpdateEnabledItems()
        {
            try
            {
                // Affected folders are those folders that have been opened 
                // already! Such folders might be empty while searching for 
                // but could contain images after finishing.
                List<NavigatorNode> affected = new List<NavigatorNode>();

                // Walk through all root nodes.
                foreach (NavigatorNode node in this.Nodes)
                {
                    // Including folders MyComputer and Favorites is not necessary 
                    // because they only support folder based items which are added 
                    // below. Folders Desktop and MyPictures instead can contain 
                    // affected files!
                    if (node.IsDesktop || node.IsPictures)
                    {
                        if (node.Nodes.Count == 0 || !node.FirstNode.IsDefaultNode)
                        {
                            affected.Add(node);
                        }
                    }

                    // Collect all currently affected folder nodes!
                    foreach (NavigatorNode folder in node.Nodes.Find(NavigatorNode.FOLDER_NODE, true))
                    {
                        if (folder.Nodes.Count == 0 || !folder.FirstNode.IsDefaultNode)
                        {
                            affected.Add(folder);
                        }
                    }
                }

                foreach (NavigatorNode node in affected)
                {
                    // Prepare a list that contains only the needed search pattern!
                    List<string> pattern = new List<string>(this.searchPattern);
                    foreach (string current in this.searchPattern)
                    {
                        if (node.Nodes.Find(current, false).Length > 0)
                        {
                            pattern.Remove(current);
                        }
                    }

                    NavigatorNode[] nodes = this.CollectFileNodes(node.NodeInfo<DirectoryInfo>(), pattern);
                    if (nodes.Length > 0)
                    {
                        node.Nodes.AddRange(nodes);
                        node.Nodes.Sort();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void RemoveDisabledItems()
        {
            try
            {
                List<NavigatorNode> affected = new List<NavigatorNode>();
                List<NavigatorNode> folders = new List<NavigatorNode>();

                // Walk through all root nodes.
                foreach (NavigatorNode node in this.Nodes)
                {
                    // Collect all nodes currently excluded.
                    foreach (string excluded in this.excludedPattern)
                    {
                        affected.AddRange(node.Nodes.Find(excluded, true));
                    }
                }

                // Now delete all affected nodes.
                foreach (NavigatorNode node in affected)
                {
                    if (!folders.Contains(node.Parent))
                    {
                        folders.Add(node.Parent);
                    }
                    node.Remove();
                }

                foreach (NavigatorNode folder in folders)
                {
                    bool hasImage = false;

                    // Search for still existing image nodes.
                    foreach (NavigatorNode file in folder.Nodes)
                    {
                        if (file.IsFileNode)
                        {
                            hasImage = true;
                            break;
                        }
                    }

                    // If an image is not longer available then restore 
                    // default folder image. Additionally, add default
                    // tree node but only if necessary!
                    if (!hasImage)
                    {
                        folder.ImageKey = folder.GetImageKey(); ;
                        folder.SelectedImageKey = folder.ImageKey;
                        if (folder.Nodes.Count <= 0)
                        {
                            folder.Nodes.Add(new DefaultNode());
                            folder.Collapse();
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void AutoExpand(NavigatorNode parent, FileSystemInfo info)
        {
            if (info != null) { this.AutoExpand(parent, info.FullName); }
        }

        private void AutoExpand(NavigatorNode parent, string source)
        {
            try
            {
                if (parent != null)
                {
                    if (!String.IsNullOrEmpty(source))
                    {
                        // Expand parent first to ensure sub-items!
                        parent.Expand();

                        if (parent.IsComputer)
                        {
                            foreach (NavigatorNode current in parent.Nodes)
                            {
                                DirectoryInfo info = current.NodeInfo<DirectoryInfo>();
                                if (info != null)
                                {
                                    string left = Path.GetPathRoot(source);
                                    string right = info.Root.ToString();
                                    if (String.Compare(left, right, true) == 0)
                                    {
                                        int start = left.Length;
                                        int count = source.Length - start;
                                        source = source.Substring(start, count);
                                        this.AutoExpand(current, source);
                                        return; // Done!
                                    }
                                }
                            }
                        }
                        else
                        {
                            // Get starting folder name...
                            string[] pieces = source.Split(new char[] { Path.DirectorySeparatorChar, Path.AltDirectorySeparatorChar });
                            if (pieces.Length > 0)
                            {
                                foreach (NavigatorNode current in parent.Nodes.Find(NavigatorNode.FOLDER_NODE, false))
                                {
                                    if (String.Compare(pieces[0], current.Text, true) == 0)
                                    {
                                        int start = Math.Min(pieces[0].Length + 1, source.Length); // Don't forget path separator!
                                        int count = Math.Max(source.Length - start, 0);
                                        source = source.Substring(start, count);
                                        this.AutoExpand(current, source);
                                        return; // Done!
                                    }
                                }

                                // If this code is reached then the given source might 
                                // contain a filename. In this case try finding that 
                                // file and select it. Otherwise select parent folder 
                                // instead.
                                FileInfo info = new FileInfo(source);
                                foreach (NavigatorNode current in parent.Nodes.Find(info.Extension, false))
                                {
                                    if (String.Compare(current.Text, source, true) == 0)
                                    {
                                        this.SelectedNode = current;
                                        return; // Done!
                                    }
                                }
                                this.SelectedNode = parent;
                                return; // Done!
                            }
                        }
                    }
                    else
                    {
                        this.SelectedNode = parent;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void JumpNext(NavigatorNode node)
        {
            try
            {
                // Calling this method might be a little bit dangerous 
                // because this function is called recursively as long 
                // as no other fitting image has been found!

                if (node == null) { return; }

                if (node.IsFileNode)
                {
                    #region Handle file nodes.

                    if (this.SelectedNode == node)
                    {
                        if (node.NextNode == null)
                        {
                            // Find next valid parent node.
                            NavigatorNode parent = node.Parent;
                            while (parent != null)
                            {
                                if (parent.NextNode != null)
                                {
                                    if (parent.NextNode.JumpAllowed)
                                    {
                                        this.JumpNext(parent.NextNode);
                                    }
                                    break; // End loop.
                                }
                                else
                                {
                                    parent = parent.Parent;
                                }
                            }
                        }
                        else if (node.NextNode.JumpAllowed)
                        {
                            this.JumpNext(node.NextNode);
                        }
                    }
                    else
                    {
                        // Show given image.
                        this.SelectedNode = node;
                        this.SelectedNode.EnsureVisible();
                    }

                    #endregion // Handle file nodes.
                }
                else
                {
                    #region Handle folder and other nodes.

                    // Ensure sub-items!
                    if (!node.IsExpanded)
                    {
                        node.Expand();
                    }

                    if (node.HasChildren)
                    {
                        this.JumpNext(node.FirstNode);
                    }
                    else if (node.NextNode == null)
                    {
                        NavigatorNode parent = node.Parent;
                        while (parent != null)
                        {
                            if (parent.NextNode != null)
                            {
                                if (parent.NextNode.JumpAllowed)
                                {
                                    this.JumpNext(parent.NextNode);
                                }
                                break; // End loop.
                            }
                            else
                            {
                                parent = parent.Parent;
                            }
                        }
                    }
                    else if (node.NextNode.JumpAllowed)
                    {
                        this.JumpNext(node.NextNode);
                    }

                    #endregion // Handle folder and other nodes.
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void JumpPrevious(NavigatorNode node)
        {
            try
            {
                // Calling this method might be a little bit dangerous 
                // because this function is called recursively as long 
                // as no other fitting image has been found!

                if (node == null) { return; }

                if (node.IsFileNode)
                {
                    #region Handle file nodes.

                    if (this.SelectedNode == node)
                    {
                        if (node.PrevNode == null)
                        {
                            NavigatorNode parent = node.Parent;
                            while (parent != null)
                            {
                                if (parent.PrevNode != null)
                                {
                                    if (parent.PrevNode.JumpAllowed)
                                    {
                                        this.JumpPrevious(parent.PrevNode);
                                    }
                                    break; // End loop.
                                }
                                else
                                {
                                    parent = parent.Parent;
                                }
                            }
                        }
                        else if (node.PrevNode.JumpAllowed)
                        {
                            this.JumpPrevious(node.PrevNode);
                        }
                    }
                    else
                    {
                        // Show given image.
                        this.SelectedNode = node;
                        this.SelectedNode.EnsureVisible();
                    }

                    #endregion // Handle file nodes.
                }
                else
                {
                    #region Handle folder and other nodes.

                    if (this.SelectedNode == node)
                    {
                        if (node.PrevNode == null)
                        {
                            NavigatorNode parent = node.Parent;
                            while (parent != null)
                            {
                                if (parent.PrevNode != null)
                                {
                                    if (parent.PrevNode.JumpAllowed)
                                    {
                                        this.JumpPrevious(parent.PrevNode);
                                    }
                                    break; // End loop.
                                }
                                else
                                {
                                    parent = parent.Parent;
                                }
                            }
                        }
                        else if (node.PrevNode.JumpAllowed)
                        {
                            this.JumpPrevious(node.PrevNode);
                        }
                    }
                    else
                    {
                        // Ensure sub-items!
                        if (!node.IsExpanded)
                        {
                            node.Expand();
                        }

                        if (node.HasChildren)
                        {
                            this.JumpPrevious(node.LastNode);
                        }
                        else if (node.PrevNode == null)
                        {
                            NavigatorNode parent = node.Parent;
                            while (parent != null)
                            {
                                if (parent.PrevNode != null)
                                {
                                    if (parent.PrevNode.JumpAllowed)
                                    {
                                        this.JumpPrevious(parent.PrevNode);
                                    }
                                    return; // Break recursion.
                                }
                                else
                                {
                                    parent = parent.Parent;
                                }
                            }
                        }
                        else if (node.PrevNode.JumpAllowed)
                        {
                            this.JumpPrevious(node.PrevNode);
                        }
                    }

                    #endregion // Handle folder and other nodes.
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private bool CanAddFolder(DirectoryInfo folder)
        {
            try
            {
                // Only existing folders should appear...
                if (folder != null && folder.Exists)
                {
                    // Checking if current folder is the "recycle bin" or not 
                    // this way is the "easy solution". Yes of course, it might 
                    // be safer to use the Shell interface instead. But such an 
                    // implementation requires a lot of work...
                    // 
                    // Yet another note is that under Windows XP the physical name 
                    // of the "recycle bin" folder is set to "RECYCLER". Windows 7 
                    // instead uses "$Recycle.Bin" instead. On the other hand, a 
                    // user might have created an own folder that include the phrase 
                    // "RECYCLE" in its name. Therefore, it seems to be a good idea 
                    // to additionally check for some attributes.
                    if (folder.Name.ToUpper().Contains("RECYCLE"))
                    {
                        if ((folder.Attributes & FileAttributes.Directory) == FileAttributes.Directory &&
                            (folder.Attributes & FileAttributes.System) == FileAttributes.System &&
                            (folder.Attributes & FileAttributes.Hidden) == FileAttributes.Hidden)
                        {
                            return false;
                        }
                        else
                        {
                            return true;
                        }
                    }
                    else
                    {
                        // Now the bad news! Obviously, it seems to be impossible 
                        // to prevent a throwing of this exception by querying some 
                        // security information. Therefore, try to access the folder 
                        // to get an exception. Not nice because it costs time but 
                        // it works...
                        try
                        {
                            FileSystemInfo[] infos = folder.GetFileSystemInfos();
                            return true;
                        }
                        catch (UnauthorizedAccessException)
                        {
                            return false;
                        }
                    }
                }
                else
                {
                    return false;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
                return false;
            }
        }

        private void ResetObservers()
        {
            // Clear old observers first.
            foreach (FileSystemWatcher watcher in this.observers)
            {
                try
                {
                    watcher.Dispose();
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }
            }
            this.observers.Clear();

            // Now reassign new observers.
            foreach (NavigatorNode root in this.Nodes)
            {
                // Observe only existing drives! Otherwise 
                // redundant events are received.
                if (root.IsComputer)
                {
                    foreach (NavigatorNode drive in root.Nodes)
                    {
                        if (!drive.IsDefaultNode)
                        {
                            this.observers.Add(this.GetObserver(drive));
                        }
                    }
                }
            }
        }

        private FileSystemWatcher GetObserver(NavigatorNode node)
        {
            string folder = node.NodeInfo<DirectoryInfo>().FullName;

            // Don't use a specific filter because otherwise 
            // folder related events are not caught.
            FileSystemWatcher result = new FileSystemWatcher(folder);

            // Initialize observer accordingly.
            result.EnableRaisingEvents = false;
            result.IncludeSubdirectories = true;
            result.SynchronizingObject = this;
            result.NotifyFilter =
                NotifyFilters.FileName |
                NotifyFilters.DirectoryName |
                NotifyFilters.LastWrite |
                NotifyFilters.Size;

            // Always assign the same observer event handlers.
            result.Created += new FileSystemEventHandler(this.OnWatcherCreated);
            result.Changed += new FileSystemEventHandler(this.OnWatcherChanged);
            result.Renamed += new RenamedEventHandler(this.OnWatcherRenamed);
            result.Deleted += new FileSystemEventHandler(this.OnWatcherDeleted);
            result.Error += new ErrorEventHandler(this.OnWatcherError);

            // Enable event receiving.
            result.EnableRaisingEvents = true;

            return result;
        }

        private void SetupItemHeight()
        {
            // Use an even item height not an odd!
            this.ItemHeight = this.Font.Height + this.Margin.Vertical;
            if ((this.ItemHeight % 2) != 0)
            {
                this.ItemHeight += 1;
            }
        }

        private void AdjustFileDropList(NavigatorNode node, bool remove)
        {
            try
            {
                if (node != null)
                {
                    FileSystemInfo info = node.NodeInfo<FileSystemInfo>();
                    if (info != null)
                    {
                        StringCollection items = Clipboard.GetFileDropList();
                        if (items != null)
                        {
                            if (remove)
                            {
                                if (items.Contains(info.FullName))
                                {
                                    items.Remove(info.FullName);
                                    if (items.Count == 0)
                                    {
                                        // BUG: May cause trouble if the clipboard contains other file types.
                                        Clipboard.Clear();
                                    }
                                    else
                                    {
                                        Clipboard.SetFileDropList(items);
                                    }
                                }
                            }
                            else
                            {
                                if (!items.Contains(info.FullName))
                                {
                                    items.Add(info.FullName);
                                    Clipboard.SetFileDropList(items);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        #endregion // Other private member implementation.

        #region Other private properties implementation.

        private Filtering Filtering
        {
            get { return Program.MainForm.Settings.Filtering; }
        }

        private Maintain Maintain
        {
            get { return Program.MainForm.Settings.Maintain; }
        }

        #endregion // Other private properties implementation.

        #region Context menu and menu item event handler implementation.

        private void ContextMenuPopupHandler(object sender, EventArgs args)
        {
            if (sender == this.menuFavorites)
            {
                this.menuItemRemoveFavorite.DefaultItem = true;
                this.menuItemRemoveFavorite.Enabled = this.CanRemoveFavorite(this.currentNode);
                this.menuItemClearAllFavorites.Enabled = this.CanClearAllFavorites(this.currentNode);
                this.menuItemExpand.Enabled = this.CanExpand(this.currentNode);
                this.menuItemCollapse.Enabled = this.CanCollapse(this.currentNode);
                this.menuItemLocate.Enabled = this.CanLocate(this.currentNode);
                this.menuItemPathOpen.Enabled = this.CanPathOpen(this.currentNode);
                this.menuItemPathCopy.Enabled = this.CanPathCopy(this.currentNode);
                this.menuItemProperties.Enabled = this.CanProperties(this.currentNode);

                this.menuFavorites.MenuItems.Clear();
                this.menuFavorites.MenuItems.AddRange(new MenuItem[] {
                    this.menuItemRemoveFavorite,
                    this.menuItemClearAllFavorites,
                    new MenuItem("-"),
                    this.menuItemExpand,
                    this.menuItemCollapse,
                    this.menuItemLocate,
                    new MenuItem("-"),
                    this.menuItemPathOpen,
                    this.menuItemPathCopy,
                    new MenuItem("-"),
                    this.menuItemProperties});
            }
            else if (sender == this.menuFolder)
            {
                this.menuItemAddToFavorites.DefaultItem = true;
                this.menuItemAddToFavorites.Enabled = this.CanAddToFavorites(this.currentNode);
                this.menuItemExpand.Enabled = this.CanExpand(this.currentNode);
                this.menuItemCollapse.Enabled = this.CanCollapse(this.currentNode);
                this.menuItemLocate.Enabled = this.CanLocate(this.currentNode);
                this.menuItemDelete.Enabled = this.CanDelete(this.currentNode);
                this.menuItemPathOpen.Enabled = this.CanPathOpen(this.currentNode);
                this.menuItemPathCopy.Enabled = this.CanPathCopy(this.currentNode);
                this.menuItemProperties.Enabled = this.CanProperties(this.currentNode);

                this.menuFolder.MenuItems.Clear();
                this.menuFolder.MenuItems.AddRange(new MenuItem[] {
                    this.menuItemAddToFavorites,
                    new MenuItem("-"),
                    this.menuItemExpand,
                    this.menuItemCollapse,
                    this.menuItemLocate,
                    this.menuItemDelete,
                    new MenuItem("-"),
                    this.menuItemPathOpen,
                    this.menuItemPathCopy,
                    new MenuItem("-"),
                    this.menuItemProperties});
            }
            else if (sender == this.menuFile)
            {
                this.menuItemFileShow.DefaultItem = true;
                this.menuItemFileShow.Enabled = this.CanFileShow(this.currentNode);
                this.menuItemFileEdit.Enabled = this.CanFileEdit(this.currentNode);
                this.menuItemExpand.Enabled = this.CanExpand(this.currentNode);
                this.menuItemCollapse.Enabled = this.CanCollapse(this.currentNode);
                this.menuItemLocate.Enabled = this.CanLocate(this.currentNode);
                this.menuItemDelete.Enabled = this.CanDelete(this.currentNode);
                this.menuItemPathOpen.Enabled = this.CanPathOpen(this.currentNode);
                this.menuItemPathCopy.Enabled = this.CanPathCopy(this.currentNode);
                this.menuItemProperties.Enabled = this.CanProperties(this.currentNode);

                this.menuItemCopyListInsert.Enabled = this.CanCopyListInsert(this.currentNode);
                this.menuItemCopyListRemove.Enabled = this.CanCopyListRemove(this.currentNode);
                this.menuItemCopyListEmpty.Enabled = this.CanCopyListEmpty(this.currentNode);
                this.menuItemCopyListContent.Enabled = this.CanCopyListContent(this.currentNode);

                this.menuItemCopyList.Enabled =
                    this.menuItemCopyListInsert.Enabled ||
                    this.menuItemCopyListRemove.Enabled ||
                    this.menuItemCopyListEmpty.Enabled ||
                    this.menuItemCopyListContent.Enabled;

                this.menuItemCopyList.MenuItems.Clear();
                this.menuItemCopyList.MenuItems.AddRange(new MenuItem[] { 
                    this.menuItemCopyListInsert, 
                    this.menuItemCopyListRemove, 
                    this.menuItemCopyListEmpty, 
                    new MenuItem("-"), 
                    this.menuItemCopyListContent});

                this.menuFile.MenuItems.Clear();
                this.menuFile.MenuItems.AddRange(new MenuItem[] {
                    this.menuItemFileShow,
                    this.menuItemFileEdit,
                    new MenuItem("-"),
                    this.menuItemExpand,
                    this.menuItemCollapse,
                    this.menuItemLocate,
                    this.menuItemDelete,
                    new MenuItem("-"),
                    this.menuItemPathOpen,
                    this.menuItemPathCopy,
                    new MenuItem("-"),
                    this.menuItemCopyList,
                    new MenuItem("-"),
                    this.menuItemProperties});
            }
        }

        private void OnMenuItemRemoveFavorite(object sender, EventArgs args)
        {
            try
            {
                this.BeginUpdate();

                if (this.currentNode != null)
                {
                    this.Maintain.RemoveFavorite(this.currentNode.NodeInfo<DirectoryInfo>());
                    this.currentNode.Remove();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.EndUpdate();
                this.currentNode = null;
            }
        }

        private void OnMenuItemClearAllFavorites(object sender, EventArgs args)
        {
            try
            {
                this.BeginUpdate();

                if (this.currentNode != null)
                {
                    while (this.currentNode.Nodes.Count > 0)
                    {
                        this.Maintain.RemoveFavorite(this.currentNode.FirstNode.NodeInfo<DirectoryInfo>());
                        this.currentNode.FirstNode.Remove();
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.EndUpdate();
                this.currentNode = null;
            }
        }

        private void OnMenuItemAddToFavorites(object sender, EventArgs args)
        {
            try
            {
                this.BeginUpdate();

                if (this.currentNode != null)
                {
                    DirectoryInfo info = this.currentNode.NodeInfo<DirectoryInfo>();
                    if (info != null)
                    {
                        foreach (NavigatorNode child in this.Nodes)
                        {
                            if (child.IsFavorites)
                            {
                                if (this.Maintain.AppendFavorite(info))
                                {
                                    // Only add a new node to the favorites if no real item is 
                                    // available. Also add a new node if other nodes than the 
                                    // default node alreasy exist.
                                    if (child.Nodes.Count == 0 || !child.FirstNode.IsDefaultNode)
                                    {
                                        FavoriteNode node = new FavoriteNode(info);
                                        node.ContextMenu = this.menuFavorites;
                                        child.Nodes.Add(node);
                                        child.Nodes.Sort();
                                    }
                                }
                                break;
                            }
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.EndUpdate();
                this.currentNode = null;
            }
        }

        private void OnMenuItemPathOpen(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode != null)
                {
                    FileSystemInfo info = this.currentNode.NodeInfo<FileSystemInfo>();
                    if (info != null)
                    {
                        if (info is DirectoryInfo)
                        {
                            Process.Start(((DirectoryInfo)info).FullName);
                        }
                        else if (info is FileInfo)
                        {
                            Process.Start(((FileInfo)info).DirectoryName);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemPathCopy(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode != null)
                {
                    FileSystemInfo info = this.currentNode.NodeInfo<FileSystemInfo>();
                    if (info != null)
                    {
                        Clipboard.SetText(String.Format("\"{0}\"", info.FullName));
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemFileShow(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode != null)
                {
                    this.SelectedNode = this.currentNode;
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemFileEdit(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode.IsFileNode)
                {
                    FileInfo info = this.currentNode.NodeInfo<FileInfo>();
                    if (info != null)
                    {
                        ProcessStartInfo startInfo = new ProcessStartInfo();
                        startInfo.UseShellExecute = true;
                        startInfo.Verb = "edit";
                        startInfo.FileName = String.Format("\"{0}\"", info.FullName);
                        Process.Start(startInfo);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemProperties(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode != null)
                {
                    FileSystemInfo info = this.currentNode.NodeInfo<FileSystemInfo>();
                    if (info != null)
                    {
                        ShellAccess.ShowPropertiesDialog(info.FullName);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemExpand(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode != null)
                {
                    this.currentNode.Expand();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemCollapse(object sender, EventArgs args)
        {
            try
            {
                if (this.currentNode != null)
                {
                    this.currentNode.Collapse();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.currentNode = null;
            }
        }

        private void OnMenuItemLocate(object sender, EventArgs args)
        {
            try
            {
                this.BeginUpdate();

                if (this.currentNode != null)
                {
                    // Walk through all root nodes.
                    foreach (NavigatorNode root in this.Nodes)
                    {
                        if (root.IsComputer)
                        {
                            if (this.currentNode.IsFolderNode || this.currentNode.IsPictures || this.currentNode.IsDesktop)
                            {
                                this.AutoExpand(root, this.currentNode.NodeInfo<FileSystemInfo>());
                            }
                            return; // Done!
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.EndUpdate();
                this.currentNode = null;
            }
        }

        private void OnMenuItemDelete(object sender, EventArgs args)
        {
            try
            {
                this.BeginUpdate();

                if (this.currentNode != null)
                {
                    // "Affected" means all tree nodes to remove 
                    // after deliting the "candidate" folder!
                    List<NavigatorNode> affected = new List<NavigatorNode>();
                    if (this.currentNode.IsFileNode)
                    {
                        FileInfo candidate = this.currentNode.NodeInfo<FileInfo>();
                        if (candidate != null)
                        {
                            foreach (NavigatorNode current in this.Nodes.Find(candidate.Extension, true))
                            {
                                FileInfo file = current.NodeInfo<FileInfo>();
                                if (file != null)
                                {
                                    if (String.Compare(candidate.FullName, file.FullName, true) == 0)
                                    {
                                        affected.Add(current);
                                    }
                                }
                            }
                        }

                        try
                        {
                            // Wow! Known how and it works...
                            FileSystem.DeleteFile(
                                candidate.FullName,
                                UIOption.AllDialogs,
                                RecycleOption.SendToRecycleBin,
                                UICancelOption.ThrowException);

                            // Now remove all affected tree nodes.
                            foreach (NavigatorNode node in affected)
                            {
                                node.Remove();

                                // Remove current file from drop list, if necessary.
                                this.AdjustFileDropList(node, true);
                            }
                        }
                        catch (OperationCanceledException)
                        {
                            // Deleting file was canceled!
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.EndUpdate();
                this.currentNode = null;
            }
        }

        private void OnMenuItemCopyListInsert(object sender, EventArgs args)
        {
            this.AdjustFileDropList(this.currentNode, false);
            this.currentNode = null;
        }

        private void OnMenuItemCopyListRemove(object sender, EventArgs args)
        {
            this.AdjustFileDropList(this.currentNode, true);
            this.currentNode = null;
        }

        private void OnMenuItemCopyListEmpty(object sender, EventArgs args)
        {
            try
            {
                if (Clipboard.ContainsFileDropList())
                {
                    // BUG: May cause trouble if the clipboard contains other file types.
                    Clipboard.Clear();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnMenuItemCopyListContent(object sender, EventArgs args)
        {
            CopyListDialog dialog = new CopyListDialog();
            dialog.ShowDialog(this);
        }

        #endregion // Context menu and menu item event handler implementation.

        #region General context menu management member function implementation.

        private bool CanRemoveFavorite(NavigatorNode node)
        {
            return node != null &&
                node.IsFavoriteNode;
        }

        private bool CanClearAllFavorites(NavigatorNode node)
        {
            return node != null &&
                node.IsFavorites &&
                this.Maintain.Favorites.Length > 0;
        }

        private bool CanAddToFavorites(NavigatorNode node)
        {
            return node != null &&
                !node.IsFavoritesSubFolder &&
                !node.IsDesktop &&
                !node.IsPictures &&
                !node.IsComputer &&
                !node.IsDriveNode &&
                !this.Maintain.HasFavorite(node.NodeInfo<DirectoryInfo>());
        }

        private bool CanPathOpen(NavigatorNode node)
        {
            return node != null &&
                !node.IsFavorites && (
                    node.IsDriveNode ||
                    node.IsFolderNode ||
                    node.IsFileNode ||
                    node.IsDesktop ||
                    node.IsPictures);
        }

        private bool CanPathCopy(NavigatorNode node)
        {
            return node != null &&
                !node.IsFavorites && (
                    node.IsDriveNode ||
                    node.IsFolderNode ||
                    node.IsFileNode ||
                    node.IsDesktop ||
                    node.IsPictures);
        }

        private bool CanFileShow(NavigatorNode node)
        {
            return node != null &&
                node.IsFileNode;
        }

        private bool CanFileEdit(NavigatorNode node)
        {
            return node != null &&
                node.IsFileNode;
        }

        private bool CanProperties(NavigatorNode node)
        {
            return node != null &&
                !node.IsFavorites && (
                    node.IsDriveNode ||
                    node.IsFolderNode ||
                    node.IsFileNode ||
                    node.IsDesktop ||
                    node.IsPictures);
        }

        private bool CanExpand(NavigatorNode node)
        {
            return node != null &&
                !node.IsExpanded &&
                node.Nodes.Count > 0;
        }

        private bool CanCollapse(NavigatorNode node)
        {
            return node != null &&
                node.IsExpanded &&
                node.Nodes.Count > 0;
        }

        private bool CanLocate(NavigatorNode node)
        {
            return node != null &&
                !node.IsFavorites &&
                !node.IsItemOfComputer && (
                    node.IsFolderNode ||
                    node.IsPictures ||
                    node.IsDesktop);
        }

        private bool CanDelete(NavigatorNode node)
        {
            try
            {
                if (node != null)
                {
                    FileInfo info = node.NodeInfo<FileInfo>();
                    if (info != null)
                    {
                        // Deleting items is only supported for files 
                        // that exist and which are not read-only.
                        return node != null &&
                            node.IsFileNode &&
                            info.Exists &&
                            !info.IsReadOnly;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private bool CanCopyListInsert(NavigatorNode node)
        {
            try
            {
                if (node != null)
                {
                    FileSystemInfo info = node.NodeInfo<FileSystemInfo>();
                    if (info != null)
                    {
                        StringCollection items = Clipboard.GetFileDropList();
                        if (items != null)
                        {
                            return !items.Contains(info.FullName);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private bool CanCopyListRemove(NavigatorNode node)
        {
            try
            {
                if (node != null)
                {
                    FileSystemInfo info = node.NodeInfo<FileSystemInfo>();
                    if (info != null)
                    {
                        StringCollection items = Clipboard.GetFileDropList();
                        if (items != null)
                        {
                            return items.Contains(info.FullName);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return false;
        }

        private bool CanCopyListEmpty(NavigatorNode node)
        {
            return Program.MainForm.ClipboardEnabled;
        }

        private bool CanCopyListContent(NavigatorNode node)
        {
            return Program.MainForm.ClipboardEnabled;
        }

        #endregion // General context menu management member function implementation.

        #region Observer event handler implementation.

        private void OnWatcherCreated(object sender, FileSystemEventArgs args)
        {
            try
            {
                // Get information about the created object!
                FileInfo info = new FileInfo(args.FullPath);

                // Check if created object is a directory.
                if ((info.Attributes & FileAttributes.Directory) == FileAttributes.Directory)
                {
                    // Try to find currently affected parent folder nodes.
                    foreach (NavigatorNode current in this.AffectedFolderNodes(info.DirectoryName))
                    {
                        if (current.Nodes.Count == 0)
                        {
                            // Add again a default node if parent was 
                            // already expanded but it didn't contain 
                            // any kind of children at that time.
                            current.Nodes.Add(new DefaultNode());
                        }
                        else if (!current.FirstNode.IsDefaultNode)
                        {
                            // Only add a new node to the parent if 
                            // it contains real items.
                            FolderNode node = new FolderNode(args.FullPath);
                            node.ContextMenu = this.menuFolder;
                            current.Nodes.Add(node);

                            // TODO: Check for item insertion functionality.
                            // At the moment it is easier to sort the complete 
                            // parent node instead of inserting the new node at 
                            // the "right" position.
                            current.Nodes.Sort();
                        }
                    }
                }
                else if (Program.InSearchPattern(info.Extension, this.searchPattern))
                {
                    // Try to find currently affected parent folder nodes.
                    foreach (NavigatorNode current in this.AffectedFolderNodes(info.DirectoryName))
                    {
                        // Check if affected file is covered by current search patterns.
                        if (Program.InSearchPattern(info.Extension, this.searchPattern))
                        {
                            if (current.Nodes.Count == 0)
                            {
                                // Add again a default node if parent was 
                                // already expanded but it didn't contain 
                                // any kind of children at that time.
                                current.Nodes.Add(new DefaultNode());
                            }
                            else if (!current.FirstNode.IsDefaultNode)
                            {
                                // Only add a new node to the parent if 
                                // it contains real items.
                                FileNode node = new FileNode(info);
                                node.ContextMenu = this.menuFile;
                                current.Nodes.Add(node);

                                // TODO: Check for item insertion functionality.
                                // At the moment it is easier to sort the complete 
                                // parent node instead of inserting the new node at 
                                // the "right" position.
                                current.Nodes.Sort();
                            }
                        }
                    }
                }
            }
            catch (UnauthorizedAccessException)
            {
                // Do nothing because it can happen and is therefore expected.
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnWatcherChanged(object sender, FileSystemEventArgs args)
        {
            try
            {
                // Get information about the changed object!
                FileInfo info = new FileInfo(args.FullPath);

                // Check if changed object is a really a file.
                if ((info.Attributes & FileAttributes.Directory) != FileAttributes.Directory)
                {
                    // Check if affected file is covered by current search patterns.
                    if (Program.InSearchPattern(info.Extension, this.searchPattern))
                    {
                        // Try to find currently affected file nodes.
                        foreach (NavigatorNode current in this.AffectedFileNodes(info.FullName))
                        {
                            // Just reassign the information of affected node.
                            current.Reassign(info.FullName);

                            // And renew current tooltip if needed.
                            if (current.IsSelected)
                            {
                                this.OnAfterSelect(new TreeViewEventArgs(current));
                            }
                        }
                    }
                }
            }
            catch (UnauthorizedAccessException)
            {
                // Do nothing because it can happen and is therefore expected.
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnWatcherRenamed(object sender, RenamedEventArgs args)
        {
            try
            {
                // Get information about the renamed object!
                FileInfo info = new FileInfo(args.FullPath);

                // Check if renamed object is a directory.
                if ((info.Attributes & FileAttributes.Directory) == FileAttributes.Directory)
                {
                    // Try to find currently affected folder nodes.
                    foreach (NavigatorNode current in this.AffectedFolderNodes(args.OldFullPath))
                    {
                        // Just reassign the information of affected node.
                        current.Reassign(info.FullName);

                        // Save current states.
                        bool reselect = current.IsSelected;

                        // TODO: Check for item insertion functionality.
                        // At the moment it is easier to sort the complete 
                        // parent node instead of inserting the new node at 
                        // the "right" position.
                        if (current.Parent != null) { current.Parent.Nodes.Sort(); }

                        // Restore previous states if necessary.
                        if (reselect) { this.SelectedNode = current; }
                    }
                }
                else
                {
                    // Try to find currently affected file nodes.
                    NavigatorNode[] affected = this.AffectedFileNodes(args.OldFullPath);
                    if (affected.Length > 0)
                    {
                        foreach (NavigatorNode current in affected)
                        {
                            // Check if affected file is covered by current search patterns.
                            if (Program.InSearchPattern(info.Extension, this.searchPattern))
                            {
                                // Just reassign the information of affected node.
                                current.Reassign(info.FullName);

                                // Save current states.
                                bool reselect = current.IsSelected;

                                // TODO: Check for item insertion functionality.
                                // At the moment it is easier to sort the complete 
                                // parent node instead of inserting the new node at 
                                // the "right" position.
                                if (current.Parent != null) { current.Parent.Nodes.Sort(); }

                                // Restore previous states if necessary.
                                if (reselect) { this.SelectedNode = current; }
                            }
                            else
                            {
                                // Currently affected file is visible at the 
                                // moment but it becomes invisible! Therefore, 
                                // remove corresponding node.
                                current.Remove();
                            }
                        }
                    }
                    else
                    {
                        // Currently affected file is invisible at the moment 
                        // but it becomes visible! Therefore, it is necessary 
                        // to add this new item to all affected folders.
                        affected = this.AffectedFolderNodes(info.DirectoryName);
                        foreach (NavigatorNode current in affected)
                        {
                            // Check if affected file is covered by current search patterns.
                            if (Program.InSearchPattern(info.Extension, this.searchPattern))
                            {
                                if (current.Nodes.Count == 0)
                                {
                                    // Add again a default node if parent was 
                                    // already expanded but it didn't contain 
                                    // any kind of children at that time.
                                    current.Nodes.Add(new DefaultNode());
                                }
                                else if (!current.FirstNode.IsDefaultNode)
                                {
                                    // Only add a new node to the parent if 
                                    // it contains real items.
                                    FileNode node = new FileNode(info);
                                    node.ContextMenu = this.menuFile;
                                    current.Nodes.Add(node);

                                    // TODO: Check for item insertion functionality.
                                    // At the moment it is easier to sort the complete 
                                    // parent node instead of inserting the new node at 
                                    // the "right" position.
                                    current.Nodes.Sort();
                                }
                            }
                        }
                    }
                }
            }
            catch (UnauthorizedAccessException)
            {
                // Do nothing because it can happen and is therefore expected.
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnWatcherDeleted(object sender, FileSystemEventArgs args)
        {
            try
            {
                // Note that a deleted file or folder does not 
                // produce a proper FileSystemInfo! Therefore, 
                // it is necessary to try to get an affected 
                // file list firstly. And only if this fails 
                // try to get an appropriated folder list.

                NavigatorNode[] affected = null;

                // Check if affected file is covered by current search patterns.
                if (Program.InSearchPattern(Path.GetExtension(args.FullPath), this.searchPattern))
                {
                    affected = this.AffectedFileNodes(args.FullPath);
                }

                // Now try to find affected folder nodes instead.
                if (affected == null || affected.Length == 0)
                {
                    affected = this.AffectedFolderNodes(args.FullPath);
                }

                // Remove all found affected tree nodes if possible.
                if (affected != null)
                {
                    foreach (NavigatorNode current in affected)
                    {
                        current.Remove();
                    }
                }
            }
            catch (UnauthorizedAccessException)
            {
                // Do nothing because it can happen and is therefore expected.
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnWatcherError(object sender, ErrorEventArgs args)
        {
            Debug.WriteLine(args.GetException());
        }

        #endregion // Observer event handler implementation.
    }

    public class NavigatorNodeCollection : ICollection<NavigatorNode>
    {
        private TreeNodeCollection collection = null;

        public NavigatorNodeCollection(TreeNodeCollection collection)
            : base()
        {
            this.collection = collection;
            foreach (TreeNode current in this.collection)
            {
                if (current is NavigatorNode)
                {
                    this.Add(current as NavigatorNode);
                }
            }
        }

        public NavigatorNode FirstNode
        {
            get
            {
                if (this.collection.Count > 0)
                {
                    return (this.collection[0] as NavigatorNode);
                }
                else
                {
                    return null;
                }
            }
        }

        public NavigatorNode[] Find(string key, bool searchAllChildren)
        {
            List<NavigatorNode> result = new List<NavigatorNode>();
            foreach (TreeNode current in this.collection.Find(key, searchAllChildren))
            {
                if (current is NavigatorNode)
                {
                    result.Add(current as NavigatorNode);
                }
            }
            return result.ToArray();
        }

        public void AddRange(NavigatorNode[] items)
        {
            this.collection.AddRange(items);
        }

        public void Sort()
        {
            try
            {
                NavigatorNode[] affected = new NavigatorNode[this.collection.Count];
                this.collection.CopyTo(affected, 0);
                Array.Sort(affected, new NavigatorNodeSorter());

                this.collection.Clear();
                this.collection.AddRange(affected);
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        #region ICollection<NavigatorNode> member implementation.

        public int Count
        {
            get { return this.collection.Count; }
        }

        public bool IsReadOnly
        {
            get { return this.collection.IsReadOnly; }
        }

        public void Add(NavigatorNode item)
        {
            this.collection.Add(item);
        }

        public void Clear()
        {
            this.collection.Clear();
        }

        public bool Contains(NavigatorNode item)
        {
            return this.collection.Contains(item);
        }

        public void CopyTo(NavigatorNode[] array, int arrayIndex)
        {
            this.collection.CopyTo(array, arrayIndex);
        }

        public bool Remove(NavigatorNode item)
        {
            this.collection.Remove(item);
            return true;
        }

        #endregion // ICollection<NavigatorNode> member implementation.

        #region IEnumerable<NavigatorNode> explicit member implementation..

        IEnumerator<NavigatorNode> IEnumerable<NavigatorNode>.GetEnumerator()
        {
            // Linq access....
            return this.collection.OfType<NavigatorNode>().GetEnumerator();
        }

        #endregion // IEnumerable<NavigatorNode> explicit member implementation..

        #region IEnumerable explicit member implementation..

        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.collection.GetEnumerator();
        }

        #endregion // IEnumerable explicit member implementation.
    }

    public class NavigatorNodeSorter : IComparer<NavigatorNode>
    {
        private SortOrder order = SortOrder.None;

        public NavigatorNodeSorter()
            : base()
        {
            this.order = SortOrder.Ascending;
        }

        public NavigatorNodeSorter(SortOrder order)
            : base()
        {
            this.order = order;
        }

        #region IComparer<NavigatorNode> members function implementation.

        public int Compare(NavigatorNode left, NavigatorNode right)
        {
            if (this.order == SortOrder.None)
            {
                return 0; // Nothing to sort.
            }
            else if (this.order == SortOrder.Ascending)
            {
                return left.CompareTo(right);
            }
            else if (this.order == SortOrder.Descending)
            {
                return -1 * left.CompareTo(right);
            }
            else
            {
                return 0; // Not an enumeration value.
            }
        }

        #endregion // IComparer<NavigatorNode> members function implementation.
    }

    public abstract class NavigatorNode : TreeNode, IComparable, IComparable<NavigatorNode>, IComparable<FileSystemInfo>
    {
        public const string DEFAULT_NODE = ".:\\DEFAULT\\:.";
        public const string FOLDER_NODE = ".:\\FOLDER\\:.";

        private NavigatorNodeCollection nodes = null;
        protected object nodeInfo = null;

        public NavigatorNode()
            : base()
        {
            this.nodes = new NavigatorNodeCollection(base.Nodes);

            this.Text = this.GetNodeText();
            this.Name = this.GetNodeName();
            this.ToolTipText = this.GetToolTipText();
            this.ImageKey = this.GetImageKey();
            this.SelectedImageKey = this.GetSelectedImageKey();
        }

        public virtual void Reassign(string text)
        {
            // Intentionally empty!
        }

        #region Abstract member declaration.

        // Returns the node's display text.
        public abstract string GetNodeText();

        // Returns the node's name to be used as search key.
        public abstract string GetNodeName();

        // Returns the node's tooltip text.
        public abstract string GetToolTipText();

        // Returns the node's standard image key.
        public abstract string GetImageKey();

        // Returns the node's selected image key.
        public abstract string GetSelectedImageKey();

        #endregion // Abstract member declaration.

        #region Data access member implementation.

        public new NavigatorNode FirstNode
        {
            get { return (base.FirstNode as NavigatorNode); }
        }

        public new NavigatorNode NextNode
        {
            get { return (base.NextNode as NavigatorNode); }
        }

        public new NavigatorNode PrevNode
        {
            get { return (base.PrevNode as NavigatorNode); }
        }

        public new NavigatorNode LastNode
        {
            get { return (base.LastNode as NavigatorNode); }
        }

        public new NavigatorNodeCollection Nodes
        {
            get { return this.nodes; }
        }

        public new NavigatorNode Parent
        {
            get { return (base.Parent as NavigatorNode); }
        }

        public new void Remove()
        {
            NavigatorNode parent = this.Parent;

            base.Remove();

            if (!this.IsDefaultNode && parent != null && parent.Nodes.Count == 0)
            {
                parent.Nodes.Add(new DefaultNode());
                parent.Collapse();

                if (parent.IsFolderNode && !parent.IsFavoriteNode)
                {
                    parent.ImageKey = parent.GetImageKey();
                    parent.SelectedImageKey = parent.GetSelectedImageKey();
                }
            }
        }

        #endregion // Data access member implementation.

        #region Node management member implementation.

        public bool IsRootNode
        {
            get
            {
                return this.IsFavorites || this.IsDesktop || this.IsPictures || this.IsComputer;
            }
        }

        public bool IsDefaultNode
        {
            get
            {
                return (this is DefaultNode);
            }
        }

        public bool IsFavorites
        {
            get
            {
                return (this is FavoritesRoot);
            }
        }

        public bool IsDesktop
        {
            get
            {
                return (this is DesktopRoot);
            }
        }

        public bool IsPictures
        {
            get
            {
                return (this is PrivatePicturesRoot || this is CommonPicturesRoot);
            }
        }

        public bool IsComputer
        {
            get
            {
                return (this is ComputerRoot);
            }
        }

        public bool IsItemOfComputer
        {
            get
            {
                // Try to find top level node.
                NavigatorNode node = this;
                while (node.Parent != null)
                {
                    node = node.Parent;
                }
                return node.IsComputer;
            }
        }

        public bool IsDriveNode
        {
            get
            {
                return (this is DriveNode);
            }
        }

        public bool IsFolderNode
        {
            get
            {
                return (this is FolderNode);
            }
        }

        public bool IsFavoriteNode
        {
            get
            {
                return (this is FavoriteNode);
            }
        }

        public bool IsFavoritesSubFolder
        {
            get
            {
                NavigatorNode node = this;
                if (node.Parent != null && !node.Parent.IsFavoriteNode)
                {
                    // Recursion until top level.
                    return node.Parent.IsFavoritesSubFolder;
                }
                else
                {
                    return this.IsFavoriteNode; //true;
                }
            }
        }

        public bool IsFileNode
        {
            get
            {
                return (this is FileNode);
            }
        }

        public bool HasChildren
        {
            get
            {
                if (this.Nodes.Count > 0)
                {
                    return !this.FirstNode.IsDefaultNode;
                }
                else
                {
                    return false;
                }
            }
        }

        public bool JumpAllowed
        {
            get
            {
                // Do not jump beyond a root node!
                return (!this.IsRootNode && !this.IsDriveNode);
            }
        }

        #endregion // Node management member implementation.

        #region Type access member implementation.

        // This code was inspired by sample on page:
        // http://en.csharp-online.net/CSharp_Coding_Solutions—Understanding_the_Overloaded_Return_Type_and_Property
        public virtual TYPE NodeInfo<TYPE>() where TYPE : class
        {
            if (typeof(DriveInfo).IsAssignableFrom(typeof(TYPE)) ||
                typeof(DirectoryInfo).IsAssignableFrom(typeof(TYPE)) ||
                typeof(FileSystemInfo).IsAssignableFrom(typeof(TYPE)) ||
                typeof(FileInfo).IsAssignableFrom(typeof(TYPE)))
            {
                return this.nodeInfo as TYPE;
            }
            else
            {
                return null;
            }
        }

        #endregion // Type access member implementation.

        #region IComparable members function implementation.

        public int CompareTo(object other)
        {
            return this.CompareTo(other as NavigatorNode);
        }

        public int CompareTo(NavigatorNode other)
        {
            int result = 0;
            if (other != null)
            {
                // Folders are "less" than files because they leading.
                if (this.IsFolderNode && other.IsFileNode)
                {
                    result = -1;
                }
                // Files are "greater" than folders because they follow.
                else if (this.IsFileNode && other.IsFolderNode)
                {
                    result = +1;
                }
                // Otherwise compare to display text.
                else if (this.IsFolderNode && other.IsFolderNode || this.IsFileNode && other.IsFileNode)
                {
                    result = String.Compare(this.Text, other.Text, true);
                    if (result == 0)
                    {
                        // In this case the additional data distinguishes the items.
                        result = this.CompareTo(other.NodeInfo<FileSystemInfo>());
                    }
                }
                else
                {
                    // REMARK: Should not happen as long as all other types remain unsorted!
                    Debug.Assert(false);
                }
            }
            return result;
        }

        public int CompareTo(FileSystemInfo other)
        {
            if (other != null)
            {
                FileSystemInfo info = this.NodeInfo<FileSystemInfo>();
                if (info != null)
                {
                    return String.Compare(info.FullName, other.FullName, true);
                }
                else
                {
                    // This instance is less that the other instacne!
                    return -1;//String.Compare(this.Text, other.FullName, true);
                }
            }
            // This instance is greater that the other instacne!
            return +1;
        }

        #endregion // IComparable members function implementation.
    }

    public class DefaultNode : NavigatorNode
    {
        #region Abstract member implementation.

        public override string GetNodeText()
        {
            return NavigatorNode.DEFAULT_NODE;
        }

        public override string GetNodeName()
        {
            return NavigatorNode.DEFAULT_NODE;
        }

        public override string GetToolTipText()
        {
            return String.Empty;
        }

        public override string GetImageKey()
        {
            return ImageNavigator.NODEICON_DEFAULT;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.
    }

    public class FavoritesRoot : NavigatorNode
    {
        public FavoritesRoot()
            : base()
        {
            this.Nodes.Add(new DefaultNode());
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            return ShellAccess.GetDisplayName(ShellAccess.CSIDL_FAVORITES);
        }

        public override string GetNodeName()
        {
            return String.Empty; // Favorites don't have a key value (yet).
        }

        public override string GetToolTipText()
        {
            return ImageNavigator.ResourceManager.GetString("TOOLTIP_OWN_FAVORITES"); ;
        }

        public override string GetImageKey()
        {
            return ImageNavigator.ROOTICON_FAVORITES;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.
    }

    public class DesktopRoot : NavigatorNode
    {
        public DesktopRoot()
            : base()
        {
            this.Nodes.Add(new DefaultNode());
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            return ShellAccess.GetDisplayName(ShellAccess.CSIDL_DESKTOP);
        }

        public override string GetNodeName()
        {
            // Treat the Desktop root node also as folder.
            return NavigatorNode.FOLDER_NODE;
        }

        public override string GetToolTipText()
        {
            DirectoryInfo info = this.NodeInfo<DirectoryInfo>();
            if (info != null)
            {
                return info.FullName;
            }
            else
            {
                return String.Empty;
            }
        }

        public override string GetImageKey()
        {
            return ImageNavigator.ROOTICON_DESKTOP;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.

        #region Type access member re-implementation.

        public override TYPE NodeInfo<TYPE>()
        {
            base.nodeInfo = null;
            try
            {
                base.nodeInfo = new DirectoryInfo(ShellAccess.GetFolderPath(ShellAccess.CSIDL_DESKTOP));
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return base.NodeInfo<TYPE>();
        }

        #endregion // Type access member re-implementation.
    }

    public class PrivatePicturesRoot : NavigatorNode
    {
        public PrivatePicturesRoot()
            : base()
        {
            this.Nodes.Add(new DefaultNode());
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            return ShellAccess.GetDisplayName(ShellAccess.CSIDL_MYPICTURES);
        }

        public override string GetNodeName()
        {
            // Treat the Pictures root node also as folder.
            return NavigatorNode.FOLDER_NODE;
        }

        public override string GetToolTipText()
        {
            DirectoryInfo info = this.NodeInfo<DirectoryInfo>();
            if (info != null)
            {
                return info.FullName;
            }
            else
            {
                return String.Empty;
            }
        }

        public override string GetImageKey()
        {
            return ImageNavigator.ROOTICON_PICTURES;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.

        #region Type access member re-implementation.

        public override TYPE NodeInfo<TYPE>()
        {
            base.nodeInfo = null;
            try
            {
                base.nodeInfo = new DirectoryInfo(ShellAccess.GetFolderPath(ShellAccess.CSIDL_MYPICTURES));
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return base.NodeInfo<TYPE>();
        }

        #endregion // Type access member re-implementation.
    }

    public class CommonPicturesRoot : NavigatorNode
    {
        public CommonPicturesRoot()
            : base()
        {
            this.Nodes.Add(new DefaultNode());
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            return ShellAccess.GetDisplayName(ShellAccess.CSIDL_COMMON_PICTURES);
        }

        public override string GetNodeName()
        {
            // Treat the Pictures root node also as folder.
            return NavigatorNode.FOLDER_NODE;
        }

        public override string GetToolTipText()
        {
            DirectoryInfo info = this.NodeInfo<DirectoryInfo>();
            if (info != null)
            {
                return info.FullName;
            }
            else
            {
                return String.Empty;
            }
        }

        public override string GetImageKey()
        {
            return ImageNavigator.ROOTICON_PICTURES;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.

        #region Type access member re-implementation.

        public override TYPE NodeInfo<TYPE>()
        {
            base.nodeInfo = null;
            try
            {
                base.nodeInfo = new DirectoryInfo(ShellAccess.GetFolderPath(ShellAccess.CSIDL_COMMON_PICTURES));
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return base.NodeInfo<TYPE>();
        }

        #endregion // Type access member re-implementation.
    }

    public class ComputerRoot : NavigatorNode
    {
        public ComputerRoot()
            : base()
        {
            this.Nodes.Add(new DefaultNode());
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            return ShellAccess.GetDisplayName(ShellAccess.CSIDL_DRIVES);
        }

        public override string GetNodeName()
        {
            return String.Empty; // Computer doesn't have a key value (yet).
        }

        public override string GetToolTipText()
        {
            return this.GetNodeText();
        }

        public override string GetImageKey()
        {
            return ImageNavigator.ROOTICON_COMPUTER;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.
    }

    public class DriveNode : NavigatorNode
    {
        private string driveName = String.Empty;

        public DriveNode(string driveName)
            : base()
        {
            this.Reassign(driveName);

            this.ImageKey = this.GetImageKey();
            this.SelectedImageKey = this.GetSelectedImageKey();

            this.Nodes.Add(new DefaultNode());
        }

        public override void Reassign(string driveName)
        {
            this.driveName = driveName;

            this.Text = this.GetNodeText();
            this.Name = this.GetNodeName();
            this.ToolTipText = this.GetToolTipText();
        }

        public bool IsDriveReady
        {
            get
            {
                DriveInfo helper = this.NodeInfo<DriveInfo>();
                if (helper != null)
                {
                    return helper.IsReady;
                }
                else
                {
                    return false;
                }
            }
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            try
            {
                DriveInfo driveInfo = this.NodeInfo<DriveInfo>();
                if (driveInfo != null)
                {
                    string label = String.Empty;
                    if (driveInfo.IsReady && !String.IsNullOrEmpty(driveInfo.VolumeLabel))
                    {
                        label = driveInfo.VolumeLabel;
                    }
                    else
                    {
                        switch (driveInfo.DriveType)
                        {
                            case DriveType.Fixed:
                                label = ImageNavigator.ResourceManager.GetString("DRIVETYPE_FIXED");
                                break;
                            case DriveType.CDRom:
                                label = ImageNavigator.ResourceManager.GetString("DRIVETYPE_CDROM");
                                break;
                            case DriveType.Removable:
                                label = ImageNavigator.ResourceManager.GetString("DRIVETYPE_REMOVABLE");
                                break;
                            case DriveType.Network:
                                label = ImageNavigator.ResourceManager.GetString("DRIVETYPE_NETWORK");
                                break;
                            case DriveType.Ram:
                                label = ImageNavigator.ResourceManager.GetString("DRIVETYPE_RAM");
                                break;
                            case DriveType.Unknown:
                            case DriveType.NoRootDirectory:
                            default:
                                label = ImageNavigator.ResourceManager.GetString("DRIVETYPE_UNKNOWN");
                                break;
                        }
                    }
                    return String.Format("{0} ({1})", label, driveInfo.Name);
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return this.driveName; // Might be empty...
        }

        public override string GetNodeName()
        {
            return String.Empty; // Drives don't have a key value (yet).
        }

        public override string GetToolTipText()
        {
            try
            {
                DriveInfo driveInfo = this.NodeInfo<DriveInfo>();
                if (driveInfo != null)
                {
                    if (driveInfo.IsReady)
                    {
                        if (!String.IsNullOrEmpty(driveInfo.VolumeLabel))
                        {
                            return String.Format(
                                ImageNavigator.ResourceManager.GetString("TOOLTIP_FREE_OF_DRIVE"),
                                driveInfo.VolumeLabel,
                                CapacityConverter.Convert(driveInfo.TotalFreeSpace),
                                CapacityConverter.Convert(driveInfo.TotalSize));
                        }
                        else
                        {
                            return String.Format(
                                ImageNavigator.ResourceManager.GetString("TOOLTIP_FREE_OF_GENERIC"),
                                CapacityConverter.Convert(driveInfo.TotalFreeSpace),
                                CapacityConverter.Convert(driveInfo.TotalSize));
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return this.driveName; // Might be empty...
        }

        public override string GetImageKey()
        {
            try
            {
                DriveInfo driveInfo = this.NodeInfo<DriveInfo>();
                if (driveInfo != null)
                {
                    switch (driveInfo.DriveType)
                    {
                        case DriveType.Fixed:
                            return ImageNavigator.DRIVEICON_FIXED;
                        case DriveType.CDRom:
                            return ImageNavigator.DRIVEICON_COMPACT;
                        case DriveType.Removable:
                            return ImageNavigator.DRIVEICON_REMOVABLE;
                        case DriveType.Network:
                            return ImageNavigator.DRIVEICON_NETWORK;
                        case DriveType.Ram:
                        case DriveType.Unknown:
                        case DriveType.NoRootDirectory:
                        default:
                            return ImageNavigator.DRIVEICON_FIXED;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return ImageNavigator.DRIVEICON_FIXED;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.

        #region Type access member re-implementation.

        public override TYPE NodeInfo<TYPE>()
        {
            base.nodeInfo = null;
            try
            {
                if (!String.IsNullOrEmpty(this.driveName))
                {
                    if (typeof(DriveInfo).IsAssignableFrom(typeof(TYPE)))
                    {
                        base.nodeInfo = new DriveInfo(this.driveName);
                    }
                    else if (typeof(FileSystemInfo).IsAssignableFrom(typeof(TYPE)) ||
                             typeof(DirectoryInfo).IsAssignableFrom(typeof(TYPE)))
                    {
                        base.nodeInfo = new DirectoryInfo(this.driveName);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return base.NodeInfo<TYPE>();
        }

        #endregion // Type access member re-implementation.
    }

    public class FolderNode : NavigatorNode
    {
        private string folderName = String.Empty;

        public FolderNode(string folderName)
            : base()
        {
            this.Reassign(folderName);

            this.ImageKey = this.GetImageKey();
            this.SelectedImageKey = this.GetSelectedImageKey();

            this.Nodes.Add(new DefaultNode());
        }

        public FolderNode(DirectoryInfo folder)
            : this(folder.FullName)
        {
        }

        public override void Reassign(string folderName)
        {
            this.folderName = folderName;

            this.Text = this.GetNodeText();
            this.Name = this.GetNodeName();
            this.ToolTipText = this.GetToolTipText();
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            string result = String.Empty;
            try
            {
                if (!String.IsNullOrEmpty(this.folderName))
                {
                    result = Path.GetFileName(this.folderName);
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return result;
        }

        public override string GetNodeName()
        {
            return NavigatorNode.FOLDER_NODE;
        }

        public override string GetToolTipText()
        {
            return this.folderName;
        }

        public override string GetImageKey()
        {
            return ImageNavigator.FOLDERICON_STANDARD;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.

        #region Type access member re-implementation.

        public override TYPE NodeInfo<TYPE>()
        {
            base.nodeInfo = null;
            try
            {
                if (!String.IsNullOrEmpty(this.folderName))
                {
                    base.nodeInfo = new DirectoryInfo(this.folderName);
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return base.NodeInfo<TYPE>();
        }

        #endregion // Type access member re-implementation.
    }

    public class FavoriteNode : FolderNode
    {
        public FavoriteNode(DirectoryInfo folder)
            : base(folder)
        {
        }

        public FavoriteNode(Favorite favorite)
            : base(new DirectoryInfo(favorite.Folder))
        {
            if (!String.IsNullOrEmpty(favorite.Name))
            {
                this.Text = favorite.Name;
            }
        }

        #region Inherited member re-implementation.

        public override string GetImageKey()
        {
            return ImageNavigator.FOLDERICON_FAVORITE;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Inherited member re-implementation.
    }

    public class FileNode : NavigatorNode
    {
        private string fileName = String.Empty;

        public FileNode(string fileName)
            : base()
        {
            this.Reassign(fileName);

            this.ImageKey = this.GetImageKey();
            this.SelectedImageKey = this.GetSelectedImageKey();
        }

        public FileNode(FileInfo file)
            : this(file.FullName)
        {
        }

        public override void Reassign(string fileName)
        {
            this.fileName = fileName;

            this.Text = this.GetNodeText();
            this.Name = this.GetNodeName();
            this.ToolTipText = this.GetToolTipText();
        }

        #region Abstract member implementation.

        public override string GetNodeText()
        {
            string result = String.Empty;
            try
            {
                if (!String.IsNullOrEmpty(this.fileName))
                {
                    result = Path.GetFileName(this.fileName);
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return result;
        }

        public override string GetNodeName()
        {
            string result = String.Empty;
            try
            {
                if (!String.IsNullOrEmpty(this.fileName))
                {
                    result = Path.GetExtension(this.fileName);
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return result;
        }

        public override string GetToolTipText()
        {
            try
            {
                FileInfo fileInfo = this.NodeInfo<FileInfo>();
                if (fileInfo != null)
                {
                    return String.Format(
                        ImageNavigator.ResourceManager.GetString("TOOLTIP_FILE_INFO"),
                        fileInfo.Name,
                        fileInfo.CreationTimeUtc.ToShortDateString(),
                        fileInfo.CreationTimeUtc.ToShortTimeString(),
                        CapacityConverter.Convert(fileInfo.Length));
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return this.GetNodeText();
        }

        public override string GetImageKey()
        {
            return ImageNavigator.FILEICON_PICTURE;
        }

        public override string GetSelectedImageKey()
        {
            return this.GetImageKey();
        }

        #endregion // Abstract member implementation.

        #region Type access member re-implementation.

        public override TYPE NodeInfo<TYPE>()
        {
            base.nodeInfo = null;
            try
            {
                if (!String.IsNullOrEmpty(this.fileName))
                {
                    base.nodeInfo = new FileInfo(this.fileName);
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return base.NodeInfo<TYPE>();
        }

        #endregion // Type access member re-implementation.
    }
}
