﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

using System.Drawing.Imaging;

using plexdata.Controls;
using plexdata.Utilities;

namespace plexdata.ImageViewer
{
    public class ImageBoxEx : ImageBox
    {
        #region Private member declaration.

        private bool loading = false;
        private Bitmap loadingImage = null;
        private Bitmap errorImage = null;

        private Color stateColorText = Color.FromArgb(0xFF, 0xFC, 0xEF, 0xDE);
        private Color stateColorDark = Color.FromArgb(0xFF, 0x45, 0x18, 0x07);
        private Color stateColorLight = Color.FromArgb(0xFF, 0xBC, 0x3F, 0x29);

        private Color errorColorText = Color.FromArgb(0xFF, 0xB2, 0xB2, 0xB2);
        private Color errorColorDark = Color.FromArgb(0xFF, 0xD3, 0xD3, 0xD3);
        private Color errorColorLight = Color.FromArgb(0xFF, 0xF4, 0xF4, 0xF4);

        private Cursor handOpenCursor = null;
        private Cursor handClosedCursor = null;

        #endregion // Private member declaration.

        public ImageBoxEx()
            : base()
        {
            // Load needed cursor resources.
            this.handOpenCursor = new Cursor(new MemoryStream(global::plexdata.ImageViewer.Properties.Resources.HandOpenCursor));
            this.handClosedCursor = new Cursor(new MemoryStream(global::plexdata.ImageViewer.Properties.Resources.HandClosedCursor));
            this.CreateErrorImage();
            this.CreateLoadingImage();
        }

        public bool Loading
        {
            get { return this.loading; }
            set
            {
                if (this.loading != value)
                {
                    this.loading = value;

                    // Redraw entire control!
                    this.Invalidate();
                    this.Update();
                }
            }
        }

        #region Overwritten event handler implementation.

        protected override void OnFontChanged(EventArgs args)
        {
            base.OnFontChanged(args);
            this.CreateErrorImage();
            this.CreateLoadingImage();
        }

        protected override void OnPaint(PaintEventArgs args)
        {
            if (this.loading && this.loadingImage != null)
            {
                // Don't draw loading state image as overlay.
                args.Graphics.Clear(this.BackColor);
                args.Graphics.DrawImage(
                    this.loadingImage,
                    new Point(
                        (this.ClientSize.Width - this.loadingImage.Width) / 2,
                        (this.ClientSize.Height - this.loadingImage.Height) / 2));
            }
            else if (this.Image == null && this.errorImage != null)
            {
                // Suppressing parent paint requires redrawing 
                // of the entire background!
                args.Graphics.Clear(this.BackColor);
                args.Graphics.DrawImage(
                    this.errorImage,
                    new Point(
                        (this.ClientSize.Width - this.errorImage.Width) / 2,
                        (this.ClientSize.Height - this.errorImage.Height) / 2));
            }
            else
            {
                base.OnPaint(args);
            }
        }

        protected override void OnErrorTextChanged(EventArgs args)
        {
            base.OnErrorTextChanged(args);
            this.CreateErrorImage();
        }

        protected override void OnZoomFactorChanged(EventArgs args)
        {
            base.OnZoomFactorChanged(args);
            this.PerformHandOpenCursor();
        }

        protected override void OnImageChanged(EventArgs args)
        {
            // Reset draw loading state because image is loaded!
            this.loading = false;

            base.OnImageChanged(args);
            this.PerformHandOpenCursor();
        }

        protected override void OnSizeChanged(EventArgs args)
        {
            base.OnSizeChanged(args);
            this.PerformHandOpenCursor();
        }

        protected override void OnMouseDown(MouseEventArgs args)
        {
            base.OnMouseDown(args);
            this.PerformHandClosedCursor();
        }

        protected override void OnMouseUp(MouseEventArgs args)
        {
            base.OnMouseUp(args);
            this.PerformHandClosedCursor();
        }

        #endregion // Overwritten event handler implementation.

        #region Private member function implementation.

        private void PerformHandOpenCursor()
        {
            if (base.Image != null)
            {
                if (base.VScroll || base.HScroll)
                {
                    this.Cursor = this.handOpenCursor;
                }
                else
                {
                    this.Cursor = Cursors.Default;
                }
            }
            else
            {
                this.Cursor = Cursors.Default;
            }
        }

        private void PerformHandClosedCursor()
        {
            if (base.Image != null)
            {
                if (base.VScroll || base.HScroll)
                {
                    if (this.Cursor == this.handOpenCursor)
                    {
                        this.Cursor = this.handClosedCursor;
                    }
                    else if (this.Cursor == this.handClosedCursor)
                    {
                        this.Cursor = this.handOpenCursor;
                    }
                }
                else
                {
                    this.Cursor = Cursors.Default;
                }
            }
            else
            {
                this.Cursor = Cursors.Default;
            }
        }

        private void CreateErrorImage()
        {
            if (this.errorImage != null)
            {
                this.errorImage.Dispose();
                this.errorImage = null;
            }

            this.errorImage = this.CreateStateImage(this.ErrorText,
                this.errorColorText, this.errorColorDark, this.errorColorLight);
        }

        private void CreateLoadingImage()
        {
            if (this.loadingImage != null)
            {
                this.loadingImage.Dispose();
                this.loadingImage = null;
            }

            this.loadingImage = this.CreateStateImage(Properties.Resources.LOAD_IMAGE_TEXT,
                this.stateColorText, this.stateColorDark, this.stateColorLight);
        }

        private Bitmap CreateStateImage(string label, Color labelColor, Color darkColor, Color lightColor)
        {
            Bitmap result = null;
            try
            {
                if (!String.IsNullOrEmpty(label))
                {
                    // Collect specific font information.
                    FontStyle style = this.Font.Style;
                    if ((this.Font.Style & FontStyle.Bold) != FontStyle.Bold &&
                        this.Font.FontFamily.IsStyleAvailable(FontStyle.Bold))
                    {
                        // A bold font looks much better on this 
                        // image instead a non-bold font.
                        style |= FontStyle.Bold;
                    }

                    using (Font font = new Font(this.Font, style))
                    {
                        // Calculate needed image size.
                        Size imageSize = new Size(60, 20);
                        using (Graphics graphics = this.CreateGraphics())
                        {
                            Size size = graphics.MeasureString(label, font).ToSize();
                            imageSize.Width += size.Width;
                            imageSize.Height += size.Height;
                        }

                        // Now create resulting bitmap and fill it.
                        result = new Bitmap(imageSize.Width, imageSize.Height);
                        using (Graphics graphics = Graphics.FromImage(result))
                        {
                            result.SetResolution(graphics.DpiX, graphics.DpiY);
                            graphics.SmoothingMode = SmoothingMode.HighQuality;
                            graphics.InterpolationMode = InterpolationMode.HighQualityBicubic;

                            #region Create graphics path for the rounded rectangle.

                            // Code below was inspired by sample on page:
                            // http://www.switchonthecode.com/tutorials/csharp-creating-rounded-rectangles-using-a-graphics-path

                            int x = 0;
                            int y = 0;
                            int width = result.Width - 1;
                            int height = result.Height - 1;
                            int radius = 6;

                            int xw = x + width;
                            int yh = y + height;
                            int xwr = xw - radius;
                            int yhr = yh - radius;
                            int xr = x + radius;
                            int yr = y + radius;
                            int r2 = radius * 2;
                            int xwr2 = xw - r2;
                            int yhr2 = yh - r2;

                            GraphicsPath path = new GraphicsPath();
                            path.StartFigure();
                            path.AddArc(x, y, r2, r2, 180, 90);     // Add rounded top/left corner.
                            path.AddLine(xr, y, xwr, y);            // Add top edge.
                            path.AddArc(xwr2, y, r2, r2, 270, 90);  // Add rounded top/right corner.
                            path.AddLine(xw, yr, xw, yhr);          // Add right edge.
                            path.AddArc(xwr2, yhr2, r2, r2, 0, 90); // Add rounded bottom/right corner.
                            path.AddLine(xwr, yh, xr, yh);          // Add bottom edge.
                            path.AddArc(x, yhr2, r2, r2, 90, 90);   // Add rounded bottom/left corner.
                            path.AddLine(x, yhr, x, yr);            // Add left edge.
                            path.CloseFigure();

                            #endregion // Create graphics path for the rounded rectangle.

                            Rectangle bounds = new Rectangle(0, 0, result.Width, result.Height);

                            // Fill rounded rectangle.
                            using (LinearGradientBrush brush = new LinearGradientBrush(
                                bounds, darkColor, lightColor, 90.0f, true))
                            {
                                // Use a bell-shaped brush with the peak 
                                // at the center of the drawing area.
                                brush.SetSigmaBellShape(0.5f, 1.0f);

                                // Draw the rounded rectangle.
                                graphics.FillPath(brush, path);
                            }

                            // Draw message text.
                            using (SolidBrush brush = new SolidBrush(labelColor))
                            {
                                StringFormat format = new StringFormat(StringFormatFlags.NoWrap);
                                format.Alignment = StringAlignment.Center;
                                format.LineAlignment = StringAlignment.Center;
                                graphics.DrawString(label, font, brush, bounds, format);
                            }
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return result;
        }

        #endregion // Private member function implementation.
    }
}
