﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.Drawing.Drawing2D;

namespace plexdata.ImageViewer
{
    public partial class SplashScreen : Form
    {
        private Color darkColor = Color.FromArgb(0xFF, 0x87, 0xAD, 0xE4);
        private Color lightColor = Color.FromArgb(0xFF, 0xCF, 0xE4, 0xFD);

        public SplashScreen()
        {
            this.InitializeComponent();

            this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);

            this.BackColor = this.TransparencyKey;

            this.BackgroundImage = this.CreateStateImage(this.Size, this.darkColor, this.lightColor);

            this.Logo = Program.ApplicationIcon.ToBitmap();

            // Title, version and copyright as well as their associated 
            // labels are language independent by design!
            this.Title = Program.ApplicationTitle;
            this.Version = "Version " + Program.ApplicationVersion;
            this.Copyright = "Copyright " + Program.ApplicationCopyright;

            this.Invalidate();
        }

        public Bitmap Logo { get; set; }

        public string Title { get; set; }

        public string Version { get; set; }

        public string Copyright { get; set; }

        protected override void OnPaint(PaintEventArgs args)
        {
            try
            {
                int left = 10;
                int top = 10;

                if (this.Logo != null)
                {
                    args.Graphics.DrawImage(this.Logo, new Point(left, top));
                    left += this.Logo.Width + 20;
                    top += (this.Logo.Height - this.Font.Height) / 2;
                }

                using (Font font = new Font(this.Font.FontFamily, 8.25f, (this.Font.Style & ~FontStyle.Bold), this.Font.Unit, this.Font.GdiCharSet, this.Font.GdiVerticalFont))
                using (Brush brush = new SolidBrush(this.ForeColor))
                {
                    if (!String.IsNullOrEmpty(this.Title))
                    {
                        args.Graphics.DrawString(this.Title, this.Font, brush, left, top);
                        top += this.Font.Height * 2;
                    }

                    if (!String.IsNullOrEmpty(this.Version))
                    {

                        args.Graphics.DrawString(this.Version, font, brush, left, top);
                        top += font.Height * 2;
                    }

                    if (!String.IsNullOrEmpty(this.Copyright))
                    {
                        args.Graphics.DrawString(this.Copyright, font, brush, left, top);
                        top += font.Height * 2;
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private Bitmap CreateStateImage(Size size, Color darkColor, Color lightColor)
        {
            Bitmap result = null;
            try
            {
                // Now create resulting bitmap and fill it.
                result = new Bitmap(size.Width, size.Height);
                using (Graphics graphics = Graphics.FromImage(result))
                {
                    result.SetResolution(graphics.DpiX, graphics.DpiY);
                    graphics.SmoothingMode = SmoothingMode.HighQuality;
                    graphics.InterpolationMode = InterpolationMode.HighQualityBicubic;

                    #region Create graphics path for the rounded rectangle.

                    // Code below was inspired by sample on page:
                    // http://www.switchonthecode.com/tutorials/csharp-creating-rounded-rectangles-using-a-graphics-path

                    int x = 0;
                    int y = 0;
                    int width = result.Width - 1;
                    int height = result.Height - 1;
                    int radius = 10;

                    int xw = x + width;
                    int yh = y + height;
                    int xwr = xw - radius;
                    int yhr = yh - radius;
                    int xr = x + radius;
                    int yr = y + radius;
                    int r2 = radius * 2;
                    int xwr2 = xw - r2;
                    int yhr2 = yh - r2;

                    GraphicsPath path = new GraphicsPath();
                    path.StartFigure();
                    path.AddArc(x, y, r2, r2, 180, 90);     // Add rounded top/left corner.
                    path.AddLine(xr, y, xwr, y);            // Add top edge.
                    path.AddArc(xwr2, y, r2, r2, 270, 90);  // Add rounded top/right corner.
                    path.AddLine(xw, yr, xw, yhr);          // Add right edge.
                    path.AddArc(xwr2, yhr2, r2, r2, 0, 90); // Add rounded bottom/right corner.
                    path.AddLine(xwr, yh, xr, yh);          // Add bottom edge.
                    path.AddArc(x, yhr2, r2, r2, 90, 90);   // Add rounded bottom/left corner.
                    path.AddLine(x, yhr, x, yr);            // Add left edge.
                    path.CloseFigure();

                    #endregion // Create graphics path for the rounded rectangle.

                    Rectangle bounds = new Rectangle(0, 0, result.Width, result.Height);

                    // Fill rounded rectangle.
                    using (LinearGradientBrush brush = new LinearGradientBrush(bounds, darkColor, lightColor, 90.0f, true))
                    {
                        // Use a bell-shaped brush with the peak 
                        // at the center of the drawing area.
                        brush.SetSigmaBellShape(0.5f, 1.0f);

                        // Draw the rounded rectangle.
                        graphics.FillPath(brush, path);
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            return result;
        }
    }
}
