﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Diagnostics;
using System.Windows.Forms;

namespace plexdata.ImageViewer
{
    public partial class SettingsPanelZooming : SettingsPanelBase
    {
        public SettingsPanelZooming(Control reference, Settings settings)
            : base()
        {
            this.InitializeComponent();

            this.Size = reference.Size;
            this.Location = reference.Location;
            this.TabIndex = reference.TabIndex;

            this.settings = settings;
            this.LoadSettings();
        }

        #region Inherited member function reimplementation.

        public override bool LoadSettings()
        {
            try
            {
                // Load zooming settings.
                this.txtErrorText.Text = this.settings.Zooming.ErrorText;
                this.chkInterpolation.Checked = this.settings.Zooming.Interpolation;
                this.numPadding.Value = this.settings.Zooming.ImagePadding;
                this.numDefault.Value = this.settings.Zooming.DefaultFactor;
                this.numMinimum.Value = this.settings.Zooming.MinimumFactor;
                this.numMaximum.Value = this.settings.Zooming.MaximumFactor;

                return true;
            }
            catch (Exception exception)
            {
                base.ErrorMessage = exception.Message;
                Debug.WriteLine(exception);
                return false;
            }
        }

        public override bool SaveSettings()
        {
            base.ErrorMessage = String.Empty;

            try
            {
                this.settings.Zooming.ErrorText = this.txtErrorText.Text;
                this.settings.Zooming.Interpolation = this.chkInterpolation.Checked;
                this.settings.Zooming.ImagePadding = this.GetValue(this.numPadding);

                int current = this.GetValue(this.numDefault);
                int minimum = this.GetValue(this.numMinimum);
                int maximum = this.GetValue(this.numMaximum);

                if (minimum >= maximum)
                {
                    base.ErrorMessage = String.Format(
                        Properties.Resources.MINMAXZOOM_ERROR_MESSAGE,
                        maximum);

                    this.numMinimum.Select(0, this.numMinimum.Text.Length);
                    this.numMinimum.Focus();

                    return false;
                }
                else if (minimum > current || current > maximum)
                {
                    base.ErrorMessage = String.Format(
                        Properties.Resources.DEFAULTZOOM_ERROR_MESSAGE,
                        minimum, maximum);

                    this.numDefault.Select(0, this.numDefault.Text.Length);
                    this.numDefault.Focus();

                    return false;
                }
                else
                {
                    this.settings.Zooming.DefaultFactor = current;
                    this.settings.Zooming.MinimumFactor = minimum;
                    this.settings.Zooming.MaximumFactor = maximum;
                }
                return true;
            }
            catch (Exception exception)
            {
                base.ErrorMessage = exception.Message;
                Debug.WriteLine(exception);
                return false;
            }
        }

        public override bool DefaultSettings()
        {
            this.settings.Zooming = new Zooming();
            return this.LoadSettings();
        }

        #endregion // Inherited member function reimplementation.

        #region Private helper function implementation.

        private int GetValue(NumericUpDown numBox)
        {
            return Convert.ToInt32(numBox.Value);
        }

        #endregion // Private helper function implementation.
    }
}
