﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;

namespace plexdata.ImageViewer
{
    public partial class SettingsPanelGeneral : SettingsPanelBase
    {
        public SettingsPanelGeneral(Control reference, Settings settings)
            : base()
        {
            this.InitializeComponent();

            this.Size = reference.Size;
            this.Location = reference.Location;
            this.TabIndex = reference.TabIndex;

            this.settings = settings;
            this.LoadSettings();
        }

        #region Inherited member function reimplementation.

        public override bool LoadSettings()
        {
            try
            {
                // Load general settings.
                this.btnForeground.Color = this.settings.General.ForeColor;
                this.btnBackground.Color = this.settings.General.BackColor;
                this.btnFont.Font = this.settings.General.Font;
                this.chkAutoFitToSize.Checked = this.settings.General.AutoAdjust;
                this.chkAutoRotate.Checked = this.settings.General.AutoRotate;

                // Load filtering settings.
                this.lstFileFormats.Clear();
                foreach (Format format in this.settings.Filtering.Formats)
                {
                    ListViewItem item = new ListViewItem(format.Name);

                    item.Tag = format;
                    item.Checked = format.Enabled;
                    item.ToolTipText = format.Display;

                    this.lstFileFormats.Items.Add(item);
                }

                ShellExtension extension = new ShellExtension();
                this.btnShellExtension.Text = extension.IsRegistered() ? Properties.Resources.BUTTON_LABEL_UNREGISTER : Properties.Resources.BUTTON_LABEL_REGISTER;

                return true;
            }
            catch (Exception exception)
            {
                base.ErrorMessage = exception.Message;
                Debug.WriteLine(exception);
                return false;
            }
        }

        public override bool SaveSettings()
        {
            base.ErrorMessage = String.Empty;

            try
            {
                this.settings.General.ForeColor = this.btnForeground.Color;
                this.settings.General.BackColor = this.btnBackground.Color;
                this.settings.General.Font = this.btnFont.Font;
                this.settings.General.AutoAdjust = this.chkAutoFitToSize.Checked;
                this.settings.General.AutoRotate = this.chkAutoRotate.Checked;

                foreach (ListViewItem item in this.lstFileFormats.Items)
                {
                    Format format = (item.Tag as Format);
                    if (format != null)
                    {
                        format.Enabled = item.Checked;
                    }
                }

                return true;
            }
            catch (Exception exception)
            {
                base.ErrorMessage = exception.Message;
                Debug.WriteLine(exception);
                return false;
            }
        }

        public override bool DefaultSettings()
        {
            this.settings.General = new General();
            this.settings.Filtering = new Filtering();
            return this.LoadSettings();
        }

        #endregion // Inherited member function reimplementation.

        #region Dialog event handler implementation.

        private void btnForeground_Click(object sender, EventArgs args)
        {
            ColorDialog dialog = new ColorDialog();
            dialog.Color = this.btnForeground.Color;
            if (DialogResult.OK == dialog.ShowDialog(this))
            {
                this.btnForeground.Color = dialog.Color;
            }
        }

        private void btnBackground_Click(object sender, EventArgs args)
        {
            ColorDialog dialog = new ColorDialog();
            dialog.Color = this.btnBackground.Color;
            if (DialogResult.OK == dialog.ShowDialog(this))
            {
                this.btnBackground.Color = dialog.Color;
            }
        }

        private void btnFont_Click(object sender, EventArgs args)
        {
            FontDialog dialog = new FontDialog();
            dialog.Font = this.btnFont.Font;
            if (DialogResult.OK == dialog.ShowDialog(this))
            {
                this.btnFont.Font = dialog.Font;
            }
        }

        private void btnShellExtension_Click(object sender, EventArgs args)
        {
            ShellExtension extension = new ShellExtension();
            if (extension.IsRegistered())
            {
                if (!extension.IsUserAdministrator())
                {
                    MessageBox.Show(this, Properties.Resources.UNREGISTER_MESSAGE_ADMIN, Properties.Resources.MESSAGE_CAPTION_NOTE, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                }
                else if (!extension.Unregister())
                {
                    MessageBox.Show(this, Properties.Resources.UNREGISTER_MESSAGE_FAILED, Properties.Resources.MESSAGE_CAPTION_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
            else
            {
                if (!extension.IsUserAdministrator())
                {
                    MessageBox.Show(this, Properties.Resources.REGISTER_MESSAGE_ADMIN, Properties.Resources.MESSAGE_CAPTION_NOTE, MessageBoxButtons.OK, MessageBoxIcon.Exclamation);
                }
                else if (!extension.Register())
                {
                    MessageBox.Show(this, Properties.Resources.REGISTER_MESSAGE_FAILED, Properties.Resources.MESSAGE_CAPTION_ERROR, MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }

            this.btnShellExtension.Text = extension.IsRegistered() ? Properties.Resources.BUTTON_LABEL_UNREGISTER : Properties.Resources.BUTTON_LABEL_REGISTER;
        }

        #endregion // Dialog event handler implementation.
    }
}
