﻿/*
 * Copyright (C)  2012  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;
using System.Collections.Generic;

namespace plexdata.ImageViewer
{
    public partial class SettingsDialog : Form
    {
        private Settings settings = null;

        public SettingsDialog()
            : this(null)
        {
        }

        public SettingsDialog(Settings settings)
            : base()
        {
            this.InitializeComponent();

            this.Icon = Program.SettingsIcon;

            this.Settings = settings; // Use a settings clone!
        }

        public Settings Settings
        {
            get { return this.settings; }
            set
            {
                if (value != null)
                {
                    this.settings = new Settings(value);
                }
                else
                {
                    this.settings = new Settings();
                }
            }
        }

        #region Dialog event handler implementation.

        private void SettingsDialogLoad(object sender, EventArgs args)
        {
            this.Controls.Remove(this.panPlaceholder);

            List<SettingsPanelBase> panels = new List<SettingsPanelBase>(
                new SettingsPanelBase[]{
                    new SettingsPanelGeneral(this.panPlaceholder, this.settings),
                    new SettingsPanelZooming(this.panPlaceholder, this.settings),
                    new SettingsPanelSlideshow(this.panPlaceholder, this.settings),
                });
            this.Controls.AddRange(panels.ToArray());

            this.lstSelector.DataSource = panels;
            this.lstSelector.ValueMember = "ValueMember";
            this.lstSelector.DisplayMember = "DisplayMember";

            // Try choosing last active settings page. If impossible 
            // then General page is chosen automatically!
            string active = this.Settings.Maintain.ActiveSettings;
            if (!String.IsNullOrEmpty(active))
            {
                foreach (SettingsPanelBase panel in panels)
                {
                    if (String.Compare(active, panel.DisplayMember, true) == 0)
                    {
                        this.lstSelector.SelectedValue = panel;
                        break;
                    }
                }
            }
        }

        private void SettingsDialogClosing(object sender, FormClosingEventArgs args)
        {
            if (this.DialogResult == DialogResult.OK)
            {
                args.Cancel = !this.ValidateSettings();

            }

            // Try saving last active settings page.
            if (!args.Cancel)
            {
                SettingsPanelBase panel = (this.lstSelector.SelectedValue as SettingsPanelBase);
                if (panel != null)
                {
                    if (this.DialogResult == DialogResult.OK)
                    {
                        // In this case current changes are taken over by the MainForm.
                        this.Settings.Maintain.ActiveSettings = panel.DisplayMember;
                    }
                    else
                    {
                        // In this case a direct manipulation is necessary!
                        Program.MainForm.Settings.Maintain.ActiveSettings = panel.DisplayMember;
                    }
                }
            }
        }

        private void SelectorListValueChanged(object sender, EventArgs args)
        {
            Control selected = (this.lstSelector.SelectedValue as Control);
            foreach (SettingsPanelBase current in this.lstSelector.Items)
            {
                if (current == selected)
                {
                    current.Show();
                }
                else
                {
                    current.Hide();
                }
            }
        }

        private void ButtonDefaultClick(object sender, EventArgs args)
        {
            SettingsPanelBase selected = (this.lstSelector.SelectedValue as SettingsPanelBase);
            if (selected != null)
            {
                selected.DefaultSettings();
            }
        }

        #endregion // Dialog event handler implementation.

        #region Private helper function implementation.

        private bool ValidateSettings()
        {
            foreach (Control current in this.Controls)
            {
                SettingsPanelBase panel = (current as SettingsPanelBase);
                if (panel != null)
                {
                    if (!panel.SaveSettings())
                    {
                        MessageBox.Show(
                            this, panel.ErrorMessage, Properties.Resources.MESSAGE_CAPTION_ERROR,
                            MessageBoxButtons.OK, MessageBoxIcon.Error);

                        // Activate page.
                        this.lstSelector.SelectedValue = panel;
                        return false;
                    }
                }
            }
            return true;
        }

        #endregion // Private helper function implementation.
    }
}
