﻿/*
 * Copyright (C)  2013  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Diagnostics;
using System.Windows.Forms;
using System.Collections.Generic;
using System.Collections.Specialized;

using plexdata.Controls;
using plexdata.Utilities;

namespace plexdata.ImageViewer
{
    public partial class CopyListDialog : Form
    {
        private ClipboardNotification clipboard = null;

        public CopyListDialog()
            : base()
        {
            this.InitializeComponent();
            this.Icon = Program.ClipboardIcon;

            this.clipboard = new ClipboardNotification(this);
            this.clipboard.ClipboardChanged += new EventHandler<EventArgs>(this.OnClipboardChanged);
        }

        protected override void OnLoad(EventArgs args)
        {
            base.OnLoad(args);
            this.LoadFileDropList();
        }

        private void OnClipboardChanged(object sender, EventArgs args)
        {
            this.LoadFileDropList();
        }

        private void OnListFilesSelectedIndexChanged(object sender, EventArgs args)
        {
            this.btnEmpty.Enabled = this.lstFiles.Items.Count > 0 && Program.MainForm.ClipboardEnabled;
            this.btnRemove.Enabled = this.lstFiles.Items.Count > 0 && this.lstFiles.SelectedIndices.Count > 0;
        }

        private void OnButtonRemoveClick(object sender, EventArgs args)
        {
            try
            {
                if (Clipboard.ContainsFileDropList())
                {
                    StringCollection drops = Clipboard.GetFileDropList();
                    if (drops != null)
                    {
                        // NOTE: Listbox content is updated as soon as the clipboard content has changed.
                        for (int index = 0; index < this.lstFiles.SelectedIndices.Count; index++)
                        {
                            FileListItem current = this.lstFiles.Items[this.lstFiles.SelectedIndices[index]] as FileListItem;
                            if (current != null && drops.Contains(current.FullName))
                            {
                                drops.Remove(current.FullName);
                            }
                        }

                        if (drops.Count == 0)
                        {
                            // BUG: May cause trouble if the clipboard contains other file types.
                            Clipboard.Clear();
                        }
                        else
                        {
                            Clipboard.SetFileDropList(drops);
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void OnButtonEmptyClick(object sender, EventArgs args)
        {
            try
            {
                if (Clipboard.ContainsFileDropList())
                {
                    // BUG: May cause trouble if the clipboard contains other file types.
                    Clipboard.Clear();
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
        }

        private void LoadFileDropList()
        {
            this.lstFiles.BeginUpdate();
            this.lstFiles.Items.Clear();
            try
            {
                // BUG: Clipboard observer also recognizes files that are copied by e.g. the Windows Explorer. 
                // At the moment there is no (proper) way to solve this issue.
                if (Clipboard.ContainsFileDropList())
                {
                    List<string> pattern = Program.MainForm.Settings.Filtering.GetSearchPattern(true);
                    foreach (string item in Clipboard.GetFileDropList())
                    {
                        if (Program.InSearchPattern(Path.GetExtension(item), pattern))
                        {
                            this.lstFiles.Items.Add(new FileListItem(item));
                        }
                    }
                }
            }
            catch (Exception exception)
            {
                Debug.WriteLine(exception);
            }
            finally
            {
                this.btnEmpty.Enabled = this.lstFiles.Items.Count > 0 && Program.MainForm.ClipboardEnabled;
                this.btnRemove.Enabled = this.lstFiles.Items.Count > 0 && this.lstFiles.SelectedIndices.Count > 0;
                this.lstFiles.EndUpdate();
            }
        }
    }
}
