﻿/*
 * Copyright (C)  2013  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;
using System.Runtime.InteropServices;

using plexdata.Utilities;

namespace plexdata.Controls
{
    public class TextEditor : Control
    {
        private const int TextIndent = 5;
        private const int ListIndent = 10;

        #region Private member variable declaration section.

        private ClipboardNotification notifier = null;
        private FixedSingleBorderRenderer borderPainter = null;

        private ToolStrip toolBar = null;
        private RichTextBoxEx textBox = null;

        private ToolStripButton tbbCut = null;
        private ToolStripButton tbbCopy = null;
        private ToolStripButton tbbPaste = null;

        private ToolStripButton tbbUndo = null;
        private ToolStripButton tbbRedo = null;

        private ToolStripButton tbbBold = null;
        private ToolStripButton tbbItalic = null;
        private ToolStripButton tbbUnderline = null;
        private ToolStripButton tbbStrike = null;

        private ToolStripButton tbbNumbers = null;
        private ToolStripButton tbbBullets = null;
        private ToolStripButton tbbSpacing = null;

        private ToolStripButton tbbJustify = null;
        private ToolStripButton tbbLeft = null;
        private ToolStripButton tbbRight = null;
        private ToolStripButton tbbCenter = null;

        private ToolStripButton tbbFont = null;
        private ToolStripButton tbbColor = null;

        #endregion // Private member variable declaration section.

        public TextEditor()
            : base()
        {
            this.InitializeComponent();
            this.borderPainter = new FixedSingleBorderRenderer(this);
        }

        #region Event section.

        public event EventHandler ModifiedChanged
        {
            add { this.textBox.ModifiedChanged += value; }
            remove { this.textBox.ModifiedChanged -= value; }
        }

        #endregion // Event section.

        #region Property section.

        public override Rectangle DisplayRectangle
        {
            get
            {
                Padding padding = this.Padding;
                Rectangle result = base.ClientRectangle;
                result.X = padding.Left;
                result.Y = padding.Top;
                result.Width -= padding.Horizontal;
                result.Height -= padding.Vertical;
                return result;
            }
        }

        protected override Size DefaultMinimumSize
        {
            get
            {
                return new Size(100, 50);
            }
        }

        protected override Size DefaultSize
        {
            get
            {
                return new Size(300, 200);
            }
        }

        protected override CreateParams CreateParams
        {
            get
            {
                // Very smart because client rectangle is also adjusted! 
                // See also: http://support.microsoft.com/kb/316574
                const int WS_BORDER = unchecked((int)0x00800000);
                const int WS_EX_STATICEDGE = unchecked((int)0x00020000);

                CreateParams createParams = base.CreateParams;
                createParams.ExStyle &= (~WS_EX_STATICEDGE);
                createParams.Style &= (~WS_BORDER);

                switch (this.BorderStyle)
                {
                    case BorderStyle.Fixed3D:
                        createParams.ExStyle |= WS_EX_STATICEDGE;
                        break;
                    case BorderStyle.FixedSingle:
                        createParams.Style |= WS_BORDER;
                        break;
                }
                return createParams;
            }
        }

        [DefaultValue(false)]
        public new bool TabStop
        {
            get { return base.TabStop; }
            set { base.TabStop = value; }
        }

        [DefaultValue(typeof(Font), "Microsoft Sans Serif, 8.25pt")]
        public new Font Font
        {
            get { return base.Font; }
            set { base.Font = value; }
        }

        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public new string Text
        {
            get
            {
                if (String.IsNullOrEmpty(this.textBox.Text.Trim()))
                {
                    return String.Empty;
                }
                else
                {
                    return this.textBox.Rtf;
                }
            }
            set
            {
                if (value != null)
                {
                    if (RtfHelper.IsRtf(value))
                    {
                        this.textBox.Rtf = value;
                    }
                    else
                    {
                        this.textBox.Text = value;
                    }
                }
                else
                {
                    this.textBox.Text = String.Empty;
                }

                // Raise text change event no matter if text has been 
                // changed or not. This is necessary to reflect state 
                // changes of Undo and Redo buttons.
                this.OnBoxTextChanged(this.textBox, EventArgs.Empty);
            }
        }

        [Browsable(false)]
        public int TextLength
        {
            get { return this.textBox.TextLength; }
        }

        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public bool Modified
        {
            get
            {
                return this.textBox.Modified;
            }
            set
            {
                this.textBox.Modified = value;
            }
        }

        [Browsable(true)]
        [DefaultValue(true)]
        [RefreshProperties(RefreshProperties.Repaint)]
        [Description("Enables or disables additional text format button.")]
        public bool AllowFormat
        {
            get
            {
                return this.allowFormat;
            }
            set
            {
                if (this.allowFormat != value)
                {
                    this.toolBar.Items.Clear();
                    if (value)
                    {
                        this.toolBar.Items.AddRange(new ToolStripItem[] {
                            this.tbbCut, this.tbbCopy, this.tbbPaste,
                            new ToolStripSeparator(), 
                            this.tbbUndo, this.tbbRedo, 
                            new ToolStripSeparator(), 
                            this.tbbBold, this.tbbItalic, this.tbbUnderline, this.tbbStrike, 
                            new ToolStripSeparator(), 
                            this.tbbNumbers, this.tbbBullets, this.tbbSpacing, 
                            new ToolStripSeparator(), 
                            this.tbbJustify, this.tbbLeft, this.tbbRight, this.tbbCenter, 
                            new ToolStripSeparator(), 
                            this.tbbFont, this.tbbColor });
                    }
                    else
                    {
                        this.toolBar.Items.AddRange(new ToolStripItem[] {
                            this.tbbCut, this.tbbCopy, this.tbbPaste,
                            new ToolStripSeparator(), 
                            this.tbbUndo, this.tbbRedo });
                    }
                    this.allowFormat = value;
                }
            }
        }
        private bool allowFormat = true;

        [Browsable(true)]
        [DefaultValue(BorderStyle.FixedSingle)]
        [RefreshProperties(RefreshProperties.Repaint)]
        [Description("Indicates whether the text editor should have a border.")]
        public BorderStyle BorderStyle
        {
            get
            {
                return this.borderStyle;
            }
            set
            {
                if (this.borderStyle != value)
                {
                    if (!Enum.IsDefined(typeof(BorderStyle), value))
                    {
                        throw new InvalidEnumArgumentException("value", (int)value, typeof(BorderStyle));
                    }

                    this.borderStyle = value;

                    this.UpdateStyles();
                }
            }
        }
        private BorderStyle borderStyle = BorderStyle.FixedSingle;

        [Browsable(true)]
        [DefaultValue(false)]
        [RefreshProperties(RefreshProperties.Repaint)]
        [Description("The integrated toolbar is hidden as soon as the text box loses the focus.")]
        public bool ToolbarHidden
        {
            get
            {
                return this.toolbarHidden;
            }
            set
            {
                if (this.toolbarHidden != value)
                {
                    this.toolbarHidden = value;

                    if (this.toolbarHidden)
                    {
                        this.HideToolbar();
                    }
                    else
                    {
                        this.ShowToolbar();
                    }
                }
            }
        }
        private bool toolbarHidden = false;

        [Browsable(true)]
        [DefaultValue(true)]
        [RefreshProperties(RefreshProperties.Repaint)]
        [Description("Indicates if lines are automatically word-wrapped for multiline edit controls.")]
        public bool WordWrap
        {
            get
            {
                return this.textBox.WordWrap;
            }
            set
            {
                this.textBox.WordWrap = value;
            }
        }

        #endregion // Property section.

        #region Public member section.

        public void SelectAll()
        {
            this.textBox.SelectAll();
        }

        #endregion // Public member section.

        #region General event handlers.

        protected override void OnFontChanged(EventArgs args)
        {
            base.OnFontChanged(args);

            FontFamily family = base.Font.FontFamily;
            this.tbbBold.Enabled = family.IsStyleAvailable(FontStyle.Bold);
            this.tbbItalic.Enabled = family.IsStyleAvailable(FontStyle.Italic);
            this.tbbUnderline.Enabled = family.IsStyleAvailable(FontStyle.Underline);
            this.tbbStrike.Enabled = family.IsStyleAvailable(FontStyle.Strikeout);

            this.textBox.Font = base.Font;
        }

        protected override void OnHandleCreated(EventArgs args)
        {
            base.OnHandleCreated(args);

            if (this.DesignMode && this.textBox.Text.Length == 0)
            {
                this.textBox.Text = "This text is only shown in the designer.";
            }
        }

        private void OnClipboardChanged(object sender, EventArgs args)
        {
            this.tbbPaste.Enabled = this.textBox.CanPaste;
        }

        protected override void OnGotFocus(EventArgs args)
        {
            base.OnGotFocus(args);
            this.textBox.Focus();
        }

        #endregion // General event handlers.

        #region Textbox related functions and event handlers.

        private void OnBoxKeyDown(object sender, KeyEventArgs args)
        {
            bool handled = true;

            // For a complete list of all supported shortcut keys see:
            // ms-help://MS.VSCC.v90/MS.MSDNQTR.v90.en/shellcc/platform/commctls/richedit/richeditcontrols/usingricheditcontrols.htm#rich_edit_shortcut_keys

            //
            // Do not handle shortcuts for Undo, Redo, Cut, Copy, and Paste!
            //

            //
            // Handling of standard shortcuts.
            //

            // Ctrl+J: Justify alignment.
            if (args.Modifiers == Keys.Control && args.KeyCode == Keys.J && this.allowFormat)
            {
                this.tbbJustify.PerformClick();
            }
            // Ctrl+L: Left alignment.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.L && this.allowFormat)
            {
                this.tbbLeft.PerformClick();
            }
            // Ctrl+R: Right alignment.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.R && this.allowFormat)
            {
                this.tbbRight.PerformClick();
            }
            // Ctrl+E: Center alignment.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.E && this.allowFormat)
            {
                this.tbbCenter.PerformClick();
            }
            // Ctrl+1: Line spacing = 1 line.  
            if (args.Modifiers == Keys.Control && args.KeyCode == Keys.D1 && this.allowFormat)
            {
                this.textBox.LineSpacing = LineSpacing.Single;
                this.ReflectParagraphSettings();
            }
            // Ctrl+2: Line spacing = 2 lines.  
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.D2 && this.allowFormat)
            {
                this.textBox.LineSpacing = LineSpacing.Double;
                this.ReflectParagraphSettings();
            }
            // Ctrl+5: Line spacing = 1.5 lines. 
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.D5 && this.allowFormat)
            {
                this.textBox.LineSpacing = LineSpacing.OneAndHalf;
                this.ReflectParagraphSettings();
            }

            //
            // Handling of extra shortcuts.
            //

            // Ctrl+B: Selection bold.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.B && this.allowFormat)
            {
                this.tbbBold.PerformClick();
            }
            // Ctrl+I: Selection italic.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.I && this.allowFormat)
            {
                this.tbbItalic.PerformClick();
            }
            // Ctrl+U: Selection underlined.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.U && this.allowFormat)
            {
                this.tbbUnderline.PerformClick();
            }
            // Ctrl+S: Selection striked out.
            else if (args.Modifiers == Keys.Control && args.KeyCode == Keys.S && this.allowFormat)
            {
                this.tbbStrike.PerformClick();
            }
            else
            {
                handled = false;
            }

            args.SuppressKeyPress = handled;
            args.Handled = handled;
        }

        private void OnBoxLinkClicked(object sender, LinkClickedEventArgs args)
        {
            Cursor cursor = this.textBox.Cursor;
            this.textBox.Cursor = Cursors.WaitCursor;
            try
            {
                Process.Start(args.LinkText);
            }
            finally
            {
                this.textBox.Cursor = cursor;
            }
        }

        private void OnBoxTextChanged(object sender, EventArgs args)
        {
            this.tbbUndo.Enabled = this.textBox.CanUndo;
            this.tbbRedo.Enabled = this.textBox.CanRedo;

            base.OnTextChanged(EventArgs.Empty);
        }

        private void OnBoxSelectionChanged(object sender, EventArgs args)
        {
            this.tbbCut.Enabled = this.textBox.CanCut;
            this.tbbCopy.Enabled = this.textBox.CanCopy;

            this.ReflectCharacterSettings();
            this.ReflectParagraphSettings();
        }

        private void OnBoxSelectionAlignmentChanged(object sender, EventArgs args)
        {
            this.ReflectParagraphSettings();
        }

        private void OnBoxLostFocus(object sender, EventArgs args)
        {
            this.HideToolbar();
        }

        private void OnBoxGotFocus(object sender, EventArgs args)
        {
            this.ShowToolbar();
        }

        #endregion // Textbox related functions and event handlers.

        #region Toolbar related functions and event handlers.

        private void OnCutClick(object sender, EventArgs args)
        {
            this.textBox.Cut();
        }

        private void OnCopyClick(object sender, EventArgs args)
        {
            this.textBox.Copy();
        }

        private void OnPasteClick(object sender, EventArgs args)
        {
            this.textBox.Paste();
        }

        private void OnUndoClick(object sender, EventArgs args)
        {
            this.textBox.Undo();
            this.tbbRedo.Enabled = this.textBox.CanRedo;
        }

        private void OnRedoClick(object sender, EventArgs args)
        {
            this.textBox.Redo();
            this.tbbUndo.Enabled = this.textBox.CanUndo;
        }

        private void OnBoldClick(object sender, EventArgs args)
        {
            this.ApplyFontStyle(FontStyle.Bold);
        }

        private void OnItalicClick(object sender, EventArgs args)
        {
            this.ApplyFontStyle(FontStyle.Italic);
        }

        private void OnUnderlineClick(object sender, EventArgs args)
        {
            this.ApplyFontStyle(FontStyle.Underline);
        }

        private void OnStrikeClick(object sender, EventArgs args)
        {
            this.ApplyFontStyle(FontStyle.Strikeout);
        }

        private void OnNumbersClick(object sender, EventArgs args)
        {
            int indent = (!this.textBox.SelectionNumber) ? TextEditor.ListIndent : TextEditor.TextIndent;
            this.textBox.BulletIndent = indent;
            this.textBox.SelectionIndent = indent;
            this.textBox.SelectionNumber = !this.textBox.SelectionNumber;
            this.ReflectParagraphSettings();
        }

        private void OnBulletsClick(object sender, EventArgs args)
        {
            int indent = (!this.textBox.SelectionBullet) ? TextEditor.ListIndent : TextEditor.TextIndent;
            this.textBox.BulletIndent = indent;
            this.textBox.SelectionIndent = indent;
            this.textBox.SelectionBullet = !this.textBox.SelectionBullet;
            this.ReflectParagraphSettings();
        }

        private void OnSpacingClick(object sender, EventArgs args)
        {
            switch (this.textBox.LineSpacing)
            {
                case LineSpacing.OneAndHalf:
                    this.textBox.LineSpacing = LineSpacing.Double;
                    break;
                case LineSpacing.Double:
                    this.textBox.LineSpacing = LineSpacing.Single;
                    break;
                case LineSpacing.Single:
                    this.textBox.LineSpacing = LineSpacing.OneAndHalf;
                    break;
            }
            this.ReflectParagraphSettings();
        }

        private void OnJustifyClick(object sender, EventArgs args)
        {
            if (this.textBox.SelectionAlignment == TextAlign.Justify)
            {
                this.textBox.SelectionAlignment = TextAlign.Default;
            }
            else
            {
                this.textBox.SelectionAlignment = TextAlign.Justify;
            }
            this.ReflectParagraphSettings();
        }

        private void OnLeftClick(object sender, EventArgs args)
        {
            if (this.textBox.SelectionAlignment == TextAlign.Left)
            {
                this.textBox.SelectionAlignment = TextAlign.Default;
            }
            else
            {
                this.textBox.SelectionAlignment = TextAlign.Left;
            }
            this.ReflectParagraphSettings();
        }

        private void OnRightClick(object sender, EventArgs args)
        {
            if (this.textBox.SelectionAlignment == TextAlign.Right)
            {
                this.textBox.SelectionAlignment = TextAlign.Default;
            }
            else
            {
                this.textBox.SelectionAlignment = TextAlign.Right;
            }
            this.ReflectParagraphSettings();
        }

        private void OnCenterClick(object sender, EventArgs args)
        {
            if (this.textBox.SelectionAlignment == TextAlign.Center)
            {
                this.textBox.SelectionAlignment = TextAlign.Default;
            }
            else
            {
                this.textBox.SelectionAlignment = TextAlign.Center;
            }
            this.ReflectParagraphSettings();
        }

        private void OnFontClick(object sender, EventArgs args)
        {
            FontDialog dialog = new FontDialog();
            dialog.Font = this.textBox.SelectionFont;
            dialog.FontMustExist = true;
            dialog.ShowApply = false;
            dialog.ShowColor = false;
            dialog.ShowHelp = false;
            if (DialogResult.OK == dialog.ShowDialog(this))
            {
                FontFamily family = dialog.Font.FontFamily;
                this.tbbBold.Enabled = family.IsStyleAvailable(FontStyle.Bold);
                this.tbbItalic.Enabled = family.IsStyleAvailable(FontStyle.Italic);
                this.tbbUnderline.Enabled = family.IsStyleAvailable(FontStyle.Underline);
                this.tbbStrike.Enabled = family.IsStyleAvailable(FontStyle.Strikeout);

                this.textBox.SelectionFont = dialog.Font;

                this.ReflectCharacterSettings();
            }
        }

        private void OnColorClick(object sender, EventArgs args)
        {
            ColorDialog dialog = new ColorDialog();
            dialog.Color = this.textBox.SelectionColor;
            dialog.AllowFullOpen = true;
            dialog.ShowHelp = false;
            dialog.SolidColorOnly = true;
            if (DialogResult.OK == dialog.ShowDialog(this))
            {
                this.textBox.SelectionColor = dialog.Color;
            }
        }

        #endregion // Toolbar related functions and event handlers.

        #region Other and private members.

        private void ReflectParagraphSettings()
        {
            // NOTE: Checked is determined to avoid/reduce flickering of the buttons.
            switch (this.textBox.SelectionAlignment)
            {
                case TextAlign.Justify:
                    if (!this.tbbJustify.Checked)
                    {
                        this.tbbJustify.Checked = true;
                        this.tbbCenter.Checked = false;
                        this.tbbLeft.Checked = false;
                        this.tbbRight.Checked = false;
                    }
                    break;
                case TextAlign.Right:
                    if (!this.tbbRight.Checked)
                    {
                        this.tbbRight.Checked = true;
                        this.tbbCenter.Checked = false;
                        this.tbbJustify.Checked = false;
                        this.tbbLeft.Checked = false;
                    }
                    break;
                case TextAlign.Center:
                    if (!this.tbbCenter.Checked)
                    {
                        this.tbbCenter.Checked = true;
                        this.tbbJustify.Checked = false;
                        this.tbbLeft.Checked = false;
                        this.tbbRight.Checked = false;
                    }
                    break;
                case TextAlign.Left:
                default:
                    if (!this.tbbLeft.Checked)
                    {
                        this.tbbLeft.Checked = true;
                        this.tbbCenter.Checked = false;
                        this.tbbJustify.Checked = false;
                        this.tbbRight.Checked = false;
                    }
                    break;
            }

            if (this.textBox.SelectionBullet && !this.tbbBullets.Checked)
            {
                this.tbbBullets.Checked = true;
            }
            else if (!this.textBox.SelectionBullet && this.tbbBullets.Checked)
            {
                this.tbbBullets.Checked = false;
            }

            if (this.textBox.SelectionNumber && !this.tbbNumbers.Checked)
            {
                this.tbbNumbers.Checked = true;
            }
            else if (!this.textBox.SelectionNumber && this.tbbNumbers.Checked)
            {
                this.tbbNumbers.Checked = false;
            }
        }

        private void ReflectCharacterSettings()
        {
            Font font = this.textBox.SelectionFont;
            if (font != null)
            {
                this.tbbBold.Checked = font.Bold;
                this.tbbItalic.Checked = font.Italic;
                this.tbbUnderline.Checked = font.Underline;
                this.tbbStrike.Checked = font.Strikeout;
            }
        }

        private void ApplyFontStyle(FontStyle style)
        {
            Font font = this.textBox.SelectionFont;
            if (font != null && font.FontFamily.IsStyleAvailable(style))
            {
                FontStyle current = font.Style;
                if ((current & style) == style)
                {
                    current &= (~style);
                }
                else
                {
                    current |= (style);
                }
                this.textBox.SelectionFont = new Font(font, current);
            }
            this.ReflectCharacterSettings();
        }

        private void HideToolbar()
        {
            if (this.toolbarHidden)
            {
                this.toolBar.Hide();
                Rectangle client = this.ClientRectangle;
                this.textBox.SetBounds(client.X, client.Y, client.Width, client.Height);
                this.textBox.Invalidate();
            }
        }

        private void ShowToolbar()
        {
            if (!this.toolBar.Visible)
            {
                this.toolBar.Show();
                Rectangle client = this.ClientRectangle;
                this.textBox.SetBounds(client.X, this.toolBar.Bottom, client.Width, client.Height - this.toolBar.Bottom);
                this.textBox.Invalidate();
            }
        }

        private void InitializeComponent()
        {
            // How to deal with different cultures is explained under link below.
            // http://stackoverflow.com/questions/1142802/how-to-use-localization-in-c-sharp
            // Further, do not use NeutralResourcesLanguage in file AssemblyInfo.cs.

            ComponentResourceManager resources = new ComponentResourceManager(typeof(TextEditor));

            this.SuspendLayout();

            this.tbbCut = new ToolStripButton(((Image)(resources.GetObject("tbbCut"))));
            this.tbbCut.Click += new EventHandler(this.OnCutClick);
            this.tbbCut.ToolTipText = resources.GetString("tttCut");

            this.tbbCopy = new ToolStripButton(((Image)(resources.GetObject("tbbCopy"))));
            this.tbbCopy.Click += new EventHandler(this.OnCopyClick);
            this.tbbCopy.ToolTipText = resources.GetString("tttCopy");

            this.tbbPaste = new ToolStripButton(((Image)(resources.GetObject("tbbPaste"))));
            this.tbbPaste.Click += new EventHandler(this.OnPasteClick);
            this.tbbPaste.ToolTipText = resources.GetString("tttPaste");

            this.tbbRedo = new ToolStripButton(((Image)(resources.GetObject("tbbRedo"))));
            this.tbbRedo.Click += new EventHandler(this.OnRedoClick);
            this.tbbRedo.ToolTipText = resources.GetString("tttRedo");

            this.tbbUndo = new ToolStripButton(((Image)(resources.GetObject("tbbUndo"))));
            this.tbbUndo.Click += new EventHandler(this.OnUndoClick);
            this.tbbUndo.ToolTipText = resources.GetString("tttUndo");

            this.tbbBold = new ToolStripButton(((Image)(resources.GetObject("tbbBold"))));
            this.tbbBold.Click += new EventHandler(this.OnBoldClick);
            this.tbbBold.ToolTipText = resources.GetString("tttBold");

            this.tbbItalic = new ToolStripButton(((Image)(resources.GetObject("tbbItalic"))));
            this.tbbItalic.Click += new EventHandler(this.OnItalicClick);
            this.tbbItalic.ToolTipText = resources.GetString("tttItalic");

            this.tbbUnderline = new ToolStripButton(((Image)(resources.GetObject("tbbUnderline"))));
            this.tbbUnderline.Click += new EventHandler(this.OnUnderlineClick);
            this.tbbUnderline.ToolTipText = resources.GetString("tttUnderline");

            this.tbbStrike = new ToolStripButton(((Image)(resources.GetObject("tbbStrikeout"))));
            this.tbbStrike.Click += new EventHandler(this.OnStrikeClick);
            this.tbbStrike.ToolTipText = resources.GetString("tttStrikeout");

            this.tbbNumbers = new ToolStripButton(((Image)(resources.GetObject("tbbNumbers"))));
            this.tbbNumbers.Click += new EventHandler(this.OnNumbersClick);
            this.tbbNumbers.ToolTipText = resources.GetString("tttNumbers");

            this.tbbBullets = new ToolStripButton(((Image)(resources.GetObject("tbbBullets"))));
            this.tbbBullets.Click += new EventHandler(this.OnBulletsClick);
            this.tbbBullets.ToolTipText = resources.GetString("tttBullets");

            this.tbbSpacing = new ToolStripButton(((Image)(resources.GetObject("tbbSpacing"))));
            this.tbbSpacing.Click += new EventHandler(this.OnSpacingClick);
            this.tbbSpacing.ToolTipText = resources.GetString("tttSpacing");

            this.tbbJustify = new ToolStripButton(((Image)(resources.GetObject("tbbJustify"))));
            this.tbbJustify.Click += new EventHandler(this.OnJustifyClick);
            this.tbbJustify.ToolTipText = resources.GetString("tttJustify");

            this.tbbLeft = new ToolStripButton(((Image)(resources.GetObject("tbbLeft"))));
            this.tbbLeft.Click += new EventHandler(this.OnLeftClick);
            this.tbbLeft.ToolTipText = resources.GetString("tttLeft");

            this.tbbRight = new ToolStripButton(((Image)(resources.GetObject("tbbRight"))));
            this.tbbRight.Click += new EventHandler(this.OnRightClick);
            this.tbbRight.ToolTipText = resources.GetString("tttRight");

            this.tbbCenter = new ToolStripButton(((Image)(resources.GetObject("tbbCenter"))));
            this.tbbCenter.Click += new EventHandler(this.OnCenterClick);
            this.tbbCenter.ToolTipText = resources.GetString("tttCenter");

            this.tbbFont = new ToolStripButton(((Image)(resources.GetObject("tbbFont"))));
            this.tbbFont.Click += new EventHandler(this.OnFontClick);
            this.tbbFont.ToolTipText = resources.GetString("tttFont");

            this.tbbColor = new ToolStripButton(((Image)(resources.GetObject("tbbColor"))));
            this.tbbColor.Click += new EventHandler(this.OnColorClick);
            this.tbbColor.ToolTipText = resources.GetString("tttColor");

            Rectangle client = this.ClientRectangle;

            this.toolBar = new ToolStrip();
            this.toolBar.Name = "toolBar";
            this.toolBar.Location = new Point(client.X, client.Y);
            this.toolBar.GripStyle = ToolStripGripStyle.Visible;
            this.toolBar.TabIndex = 1;
            this.toolBar.Text = String.Empty;
            this.toolBar.GripStyle = ToolStripGripStyle.Hidden;
            this.toolBar.Padding = new Padding(3, 0, 1, 0);

            this.toolBar.Items.AddRange(new ToolStripItem[] {
                this.tbbCut, this.tbbCopy, this.tbbPaste,
                new ToolStripSeparator(), 
                this.tbbUndo, this.tbbRedo, 
                new ToolStripSeparator(), 
                this.tbbBold, this.tbbItalic, this.tbbUnderline, this.tbbStrike, 
                new ToolStripSeparator(), 
                this.tbbNumbers, this.tbbBullets, this.tbbSpacing, 
                new ToolStripSeparator(), 
                this.tbbJustify, this.tbbLeft, this.tbbRight, this.tbbCenter, 
                new ToolStripSeparator(), 
                this.tbbFont, this.tbbColor });

            ToolStripProfessionalRenderer renderer = new ToolStripProfessionalRenderer(new ProfessionalColorTable());
            renderer.RoundedEdges = false;
            this.toolBar.Renderer = renderer;

            this.textBox = new RichTextBoxEx();
            this.textBox.Name = "textBox";
            this.textBox.Location = new Point(client.X, this.toolBar.Bottom);
            this.textBox.Size = new Size(client.Width, client.Height - this.toolBar.Bottom);
            this.textBox.BorderStyle = BorderStyle.None;
            this.textBox.Anchor = AnchorStyles.Top | AnchorStyles.Bottom | AnchorStyles.Left | AnchorStyles.Right;
            this.textBox.AcceptsTab = true;
            this.textBox.WordWrap = true;
            this.textBox.TabIndex = 0;
            this.textBox.Text = String.Empty;
            this.textBox.SelectionIndent = TextEditor.TextIndent;
            this.textBox.SelectionRightIndent = TextEditor.TextIndent;
            this.textBox.DetectUrls = true;
            // Do not disable default shortcut handling!
            this.textBox.ShortcutsEnabled = true;
            this.textBox.SpaceBefore = TextIndent;
            this.textBox.LineSpacing = LineSpacing.Default;
            this.textBox.KeyDown += new KeyEventHandler(this.OnBoxKeyDown);
            this.textBox.LinkClicked += new LinkClickedEventHandler(this.OnBoxLinkClicked);
            this.textBox.TextChanged += new EventHandler(this.OnBoxTextChanged);
            this.textBox.SelectionChanged += new EventHandler(this.OnBoxSelectionChanged);
            this.textBox.GotFocus += new EventHandler(this.OnBoxGotFocus);
            this.textBox.LostFocus += new EventHandler(this.OnBoxLostFocus);
            this.textBox.SelectionAlignmentChanged += new EventHandler<EventArgs>(this.OnBoxSelectionAlignmentChanged);

            this.textBox.ClearUndo();
            this.tbbCut.Enabled = this.textBox.CanCut;
            this.tbbCopy.Enabled = this.textBox.CanCopy;
            this.tbbPaste.Enabled = this.textBox.CanPaste;
            this.tbbRedo.Enabled = this.textBox.CanRedo;
            this.tbbUndo.Enabled = this.textBox.CanUndo;

            this.ReflectParagraphSettings();

            this.TabStop = false;
            this.Font = new Font("Microsoft Sans Serif", 8.25f);
            this.Controls.Add(this.textBox);
            this.Controls.Add(this.toolBar);

            this.notifier = new ClipboardNotification(this);
            this.notifier.ClipboardChanged += new EventHandler<EventArgs>(this.OnClipboardChanged);

            this.ResumeLayout(false);
        }

        #endregion // Other and private members.
    }

    public enum TextAlign { Default = Left, Left = 1, Right = 2, Center = 3, Justify = 4 }

    public enum LineSpacing { Default = Single, Single = 0, OneAndHalf = 1, Double = 2 }

    public class RichTextBoxEx : RichTextBox
    {
        // NOTE: The handling of message WM_NCCALCSIZE did not satisfyingly solve the "border" problem!

        // Base code of this class came from:
        // http://geekswithblogs.net/pvidler/archive/2003/10/14/182.aspx

        // Find additional information under:
        // ms-help://MS.VSCC.v90/MS.MSDNQTR.v90.en/shellcc/platform/commctls/richedit/richeditcontrols/richeditcontrolreference/richeditstructures/paraformat2.htm

        // TODO: Support different number list styles.

        public event EventHandler<EventArgs> SelectionAlignmentChanged;

        public RichTextBoxEx()
            : base()
        {
            // NOTE: Double Buffering does not have any effect on flicker free resizing if images are included!
        }

        public bool CanCut
        {
            get { return this.SelectionLength > 0; }
        }

        public bool CanCopy
        {
            get { return this.SelectionLength > 0; }
        }

        public new bool CanPaste
        {
            get
            {
                IDataObject objects = Clipboard.GetDataObject();
                if (objects != null)
                {
                    foreach (string format in objects.GetFormats())
                    {
                        if (base.CanPaste(DataFormats.GetFormat(format)))
                        {
                            return true;
                        }
                    }
                }
                return false;
            }
        }

        public bool SelectionNumber
        {
            get
            {
                PARAFORMAT fmt = new PARAFORMAT();
                fmt.cbSize = Marshal.SizeOf(fmt);
                SendMessage(this.Handle, EM_GETPARAFORMAT, SCF_SELECTION, ref fmt);
                if ((fmt.dwMask & PFM_NUMBERING) == 0)
                {
                    return false;
                }
                else
                {
                    return (BulletType)fmt.wNumbering == BulletType.Number;
                }
            }
            set
            {
                if (this.SelectionNumber != value)
                {
                    PARAFORMAT fmt = new PARAFORMAT();
                    fmt.cbSize = Marshal.SizeOf(fmt);
                    fmt.dwMask = unchecked((uint)(
                        PFM_NUMBERING | PFM_NUMBERINGSTART | PFM_NUMBERINGSTYLE |
                        PFM_NUMBERINGTAB | PFM_OFFSET | PFM_STARTINDENT | PFM_RIGHTINDENT));

                    if (!value)
                    {
                        fmt.wNumbering = 0;
                        fmt.dxOffset = 0;
                    }
                    else
                    {
                        fmt.wNumbering = (short)BulletType.Number;
                        fmt.wNumberingStyle = (short)BulletStyle.Period;
                        fmt.wNumberingStart = 1; // Start at 1.

                        // All values needed in twips!
                        fmt.wNumberingTab = (short)(this.BulletIndent * 20);
                        fmt.dxOffset = fmt.wNumberingTab;
                        fmt.dxStartIndent = this.SelectionIndent * 20;
                        fmt.dxRightIndent = this.SelectionRightIndent * 20;
                    }
                    SendMessage(this.Handle, EM_SETPARAFORMAT, 0, ref fmt);
                }
            }
        }

        // Space before in points.
        public int SpaceBefore
        {
            get
            {
                PARAFORMAT fmt = new PARAFORMAT();
                fmt.cbSize = Marshal.SizeOf(fmt);

                SendMessage(this.Handle, EM_GETPARAFORMAT, SCF_SELECTION, ref fmt);

                if ((fmt.dwMask & PFM_SPACEBEFORE) == 0)
                {
                    return 0;
                }
                else
                {
                    return fmt.dySpaceBefore;
                }
            }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException("SpaceBefore");
                }
                else if (this.SpaceBefore != value * 20)
                {
                    PARAFORMAT fmt = new PARAFORMAT();
                    fmt.cbSize = Marshal.SizeOf(fmt);
                    fmt.dwMask = PFM_SPACEBEFORE;
                    fmt.dySpaceBefore = value * 20; // Needed in twips!
                    SendMessage(this.Handle, EM_SETPARAFORMAT, SCF_SELECTION, ref fmt);
                }
            }
        }

        // Space after in points.
        public int SpaceAfter
        {
            get
            {
                PARAFORMAT fmt = new PARAFORMAT();
                fmt.cbSize = Marshal.SizeOf(fmt);

                SendMessage(this.Handle, EM_GETPARAFORMAT, SCF_SELECTION, ref fmt);

                if ((fmt.dwMask & PFM_SPACEAFTER) == 0)
                {
                    return 0;
                }
                else
                {
                    return fmt.dySpaceAfter;
                }
            }
            set
            {
                if (value < 0)
                {
                    throw new ArgumentOutOfRangeException("SpaceAfter");
                }
                else if (this.SpaceAfter != value * 20)
                {
                    PARAFORMAT fmt = new PARAFORMAT();
                    fmt.cbSize = Marshal.SizeOf(fmt);
                    fmt.dwMask = PFM_SPACEAFTER;
                    fmt.dySpaceAfter = value * 20; // Needed in twips!
                    SendMessage(this.Handle, EM_SETPARAFORMAT, SCF_SELECTION, ref fmt);
                }
            }
        }

        public LineSpacing LineSpacing
        {
            get
            {
                PARAFORMAT fmt = new PARAFORMAT();
                fmt.cbSize = Marshal.SizeOf(fmt);

                SendMessage(this.Handle, EM_GETPARAFORMAT, SCF_SELECTION, ref fmt);

                if ((fmt.dwMask & PFM_LINESPACING) == 0)
                {
                    return LineSpacing.Default;
                }
                else
                {
                    return (LineSpacing)fmt.bLineSpacingRule;
                }
            }
            set
            {
                if (this.LineSpacing != value)
                {
                    PARAFORMAT fmt = new PARAFORMAT();
                    fmt.cbSize = Marshal.SizeOf(fmt);
                    fmt.dwMask = PFM_LINESPACING;

                    switch (value)
                    {
                        case LineSpacing.OneAndHalf:
                            fmt.dyLineSpacing = 30; // Needed in twips!
                            break;
                        case LineSpacing.Double:
                            fmt.dyLineSpacing = 40; // Needed in twips!
                            break;
                        case LineSpacing.Single:
                        default:
                            fmt.dyLineSpacing = 20; // Needed in twips!
                            break;
                    }

                    fmt.bLineSpacingRule = (byte)value;

                    SendMessage(this.Handle, EM_SETPARAFORMAT, SCF_SELECTION, ref fmt);
                }
            }
        }

        public new TextAlign SelectionAlignment
        {
            get
            {
                PARAFORMAT fmt = new PARAFORMAT();
                fmt.cbSize = Marshal.SizeOf(fmt);

                SendMessage(this.Handle, EM_GETPARAFORMAT, SCF_SELECTION, ref fmt);

                if ((fmt.dwMask & PFM_ALIGNMENT) == 0)
                {
                    return TextAlign.Default;
                }
                else
                {
                    return (TextAlign)fmt.wAlignment;
                }
            }
            set
            {
                if (this.SelectionAlignment != value)
                {
                    PARAFORMAT fmt = new PARAFORMAT();
                    fmt.cbSize = Marshal.SizeOf(fmt);
                    fmt.dwMask = PFM_ALIGNMENT;
                    fmt.wAlignment = (short)value;

                    SendMessage(this.Handle, EM_SETPARAFORMAT, SCF_SELECTION, ref fmt);

                    if (this.SelectionAlignmentChanged != null)
                    {
                        this.SelectionAlignmentChanged(this, EventArgs.Empty);
                    }
                }
            }
        }

        public void BeginUpdate()
        {
            // Deal with nested calls.
            ++updating;

            if (updating > 1) { return; }

            // Prevent the control from raising any events.
            oldEventMask = SendMessage(this.Handle, EM_SETEVENTMASK, 0, 0);

            // Prevent the control from redrawing itself.
            SendMessage(this.Handle, WM_SETREDRAW, 0, 0);
        }

        public void EndUpdate()
        {
            // Deal with nested calls.
            --updating;

            if (updating > 0) { return; }

            // Allow the control to redraw itself.
            SendMessage(this.Handle, WM_SETREDRAW, 1, 0);

            // Allow the control to raise event messages.
            SendMessage(this.Handle, EM_SETEVENTMASK, 0, oldEventMask);

            this.Invalidate();
        }

        protected override void OnHandleCreated(EventArgs args)
        {
            base.OnHandleCreated(args);

            // Enable support for justification.
            SendMessage(this.Handle, EM_SETTYPOGRAPHYOPTIONS,
                TO_ADVANCEDTYPOGRAPHY, TO_ADVANCEDTYPOGRAPHY);
        }

        private enum BulletType { Normal = 1, Number = 2, LowerCaseLetter = 3, UpperCaseLetter = 4, LowerCaseRoman = 5, UpperCaseRoman = 6 }

        private enum BulletStyle { RightParenthesis = 0x000, DoubleParenthesis = 0x100, Period = 0x200, Plain = 0x300, NoNumber = 0x400 }

        private int updating = 0;

        private int oldEventMask = 0;

        #region Win 32 related functions and declarations.

        // Constants from the Platform SDK.
        private const int EM_SETEVENTMASK = 1073;
        private const int EM_GETPARAFORMAT = 1085;
        private const int EM_SETPARAFORMAT = 1095;
        private const int EM_SETTYPOGRAPHYOPTIONS = 1226;
        private const int WM_SETREDRAW = 11;
        private const int TO_ADVANCEDTYPOGRAPHY = 1;
        private const int SCF_SELECTION = 1;
        private const int PFM_ALIGNMENT = 0x00000008;
        private const int PFM_SPACEBEFORE = 0x00000040;
        private const int PFM_SPACEAFTER = 0x00000080;
        private const int PFM_LINESPACING = 0x00000100;
        private const int PFM_STARTINDENT = 0x00000001;
        private const int PFM_RIGHTINDENT = 0x00000002;
        private const int PFM_OFFSETINDENT = unchecked((int)0x80000000);
        private const int PFM_NUMBERING = 0x00000020;
        private const int PFM_OFFSET = 0x00000004;
        private const int PFM_NUMBERINGSTYLE = 0x00002000;
        private const int PFM_NUMBERINGTAB = 0x00004000;
        private const int PFM_NUMBERINGSTART = 0x00008000;

        // It makes no difference if we use PARAFORMAT or
        // PARAFORMAT2 here, so I have opted for PARAFORMAT2.
        [StructLayout(LayoutKind.Sequential)]
        private struct PARAFORMAT
        {
            public int cbSize;
            public uint dwMask;
            public short wNumbering;
            public short wReserved;
            public int dxStartIndent;
            public int dxRightIndent;
            public int dxOffset;
            public short wAlignment;
            public short cTabCount;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 32)]
            public int[] rgxTabs;

            // PARAFORMAT2 from here onwards.
            public int dySpaceBefore;
            public int dySpaceAfter;
            public int dyLineSpacing;
            public short sStyle;
            public byte bLineSpacingRule;
            public byte bOutlineLevel;
            public short wShadingWeight;
            public short wShadingStyle;
            public short wNumberingStart;
            public short wNumberingStyle;
            public short wNumberingTab;
            public short wBorderSpace;
            public short wBorderWidth;
            public short wBorders; // This really means a paragaph's border!
        }

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern int SendMessage(IntPtr hWnd, int message, int wParam, int lParam);

        [DllImport("user32.dll", CharSet = CharSet.Auto, SetLastError = true)]
        private static extern int SendMessage(IntPtr hWnd, int message, int wParam, ref PARAFORMAT lParam);

        #endregion // Win 32 related functions and declarations.
    }
}
