﻿/*
 * Copyright (C)  2011  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;
using System.Drawing.Imaging;
using System.Drawing.Drawing2D;
using System.Security.Permissions;
using System.Runtime.InteropServices;

namespace plexdata.Controls
{
    /// <summary>
    /// Scrollable and zoomable image view box.
    /// </summary>
    /// <remarks>
    /// The class <i>ImageBox</i>, derived from class <see cref="Control"/>, 
    /// represents a scrollable and zoomable view box that is intended to be 
    /// used to display any kind of image.
    /// <para>
    /// <b><i>Keyboard Actions</i></b>
    /// </para>
    /// <para>
    /// This control supports various keyboard keys to accomplish the work. 
    /// See lists below which keyboard keys and shortcuts are available.
    /// </para>
    /// <para>
    /// <list type="table">
    /// <listheader>
    ///     <term>
    ///     Keys
    ///     </term>
    ///     <description>
    ///     Description
    ///     </description>
    /// </listheader>
    /// <item>
    ///     <term>
    ///     HOME
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content to the very first horizontal 
    ///     position whereby current vertical position is kept. If the <b>SHIFT</b> 
    ///     key is additionally pressed then the visible content is scrolled 
    ///     to the very first vertical position. Current horizontal position 
    ///     is kept in this case. If the <b>CTRL</b> key is pressed instead 
    ///     then the visible content is scrolled to the upper left corner.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     END
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content to the very last horizontal 
    ///     position whereby current vertical position is kept. If the <b>SHIFT</b> 
    ///     key is additionally pressed then the visible content is scrolled 
    ///     to the very last vertical position. Current horizontal position 
    ///     is kept in this case. If the <b>CTRL</b> key is pressed instead 
    ///     then the visible content is scrolled to the lower right corner.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     UP
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content a small amount upward. If the 
    ///     <b>CTRL</b> key is additionally pressed then the visible content is 
    ///     scrolled one page up.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     DOWN
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content a small amount downward. If the 
    ///     <b>CTRL</b> key is additionally pressed then the visible content is 
    ///     scrolled one page down.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     LEFT
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content a small amount to the left. If 
    ///     the <b>CTRL</b> key is additionally pressed then the visible content 
    ///     is scrolled one page to the left.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     RIGHT
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content a small amount to the right. If 
    ///     the <b>CTRL</b> key is additionally pressed then the visible content 
    ///     is scrolled one page to the right.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     PAGE UP
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content one page upward. If the <b>CTRL</b> 
    ///     key is additionally pressed then the visible content is scrolled one 
    ///     page to the left instead.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     PAGE DOWN
    ///     </term>
    ///     <description>
    ///     This key scrolls the visible content one page downward. If the <b>CTRL</b> 
    ///     key is additionally pressed then the visible content is scrolled one 
    ///     page to the right instead.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     CTRL+PLUS
    ///     </term>
    ///     <description>
    ///     This keyboard key magnifies currently visible content. In this case 
    ///     the standard zoom-in behavior is performed.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     CTRL+MINUS
    ///     </term>
    ///     <description>
    ///     This keyboard key downsizes currently visible content. In this case 
    ///     the standard zoom-out behavior is performed.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     CTRL+ZERO
    ///     </term>
    ///     <description>
    ///     These keyboard key returns to the default magnification which in 
    ///     fact depends on current default zoom factor.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     TAB
    ///     </term>
    ///     <description>
    ///     This key activates the next selectable control in the tab order. If the 
    ///     <b>SHIFT</b> key is additionally pressed then the previous control is 
    ///     selected instead.
    ///     </description>
    /// </item>
    /// </list>
    /// </para>
    /// <para>
    /// <b><i>Mouse Actions</i></b>
    /// </para>
    /// <para>
    /// Furthermore, this control supports various mouse actions to facilitate the 
    /// work. See list below which mouse actions are available at all.
    /// </para>
    /// <para>
    /// <list type="table">
    /// <listheader>
    ///     <term>
    ///     Actions
    ///     </term>
    ///     <description>
    ///     Description
    ///     </description>
    /// </listheader>
    /// <item>
    ///     <term>
    ///     Dragging
    ///     </term>
    ///     <description>
    ///     The currently visible content can be dragged with the mouse. To do 
    ///     this press and keep pressed the left mouse button. Then drag the 
    ///     mouse to scroll currently visible content. After finishing simply 
    ///     release left mouse button.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     Scrolling
    ///     </term>
    ///     <description>
    ///     The currently visible content can be scrolled using the mouse. To do 
    ///     scroll up simply rotate the mouse wheel forward. To scroll down just 
    ///     rotate the mouse wheel backward. If the <b>SHIFT</b> key is additionally 
    ///     pressed on the keyboard then a mouse wheel forward rotation scrolls 
    ///     current visible content to the left. A mouse wheel backward rotation 
    ///     scrolls current visible content to the right instead.
    ///     </description>
    /// </item>
    /// <item>
    ///     <term>
    ///     Zooming 
    ///     </term>
    ///     <description>
    ///     The currently visible content can be zoomed in or out using the mouse. 
    ///     To zoom in simply press and keep pressed keyboard key <b>CTRL</b> and 
    ///     rotate the mouse wheel forward. To zoom out also press and keep pressed 
    ///     keyboard key <b>CTRL</b> but rotate the mouse wheel backward. 
    ///     </description>
    /// </item>
    /// </list>
    /// </para>
    /// </remarks>
    public class ImageBox : ScrollableControl
    {
        #region Private member variable declaration.

        /// <summary>
        /// Indicates current mouse dragging positon.
        /// </summary>
        private Point dragOffset = Point.Empty;

        #endregion // Private member variable declaration.

        /// <summary>
        /// Initializes a new instance of the <i>ImageBox</i> class 
        /// with default settings.
        /// </summary>
        public ImageBox()
            : base()
        {
            // Set some window styles.
            this.SetStyle(ControlStyles.Selectable, true);
            this.SetStyle(ControlStyles.ResizeRedraw, true);
            this.SetStyle(ControlStyles.UserPaint, true);
            this.SetStyle(ControlStyles.DoubleBuffer, true);
            this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            this.UpdateStyles();

            // Enable auto scroll mode.
            this.AutoScroll = true;
        }

        #region Property BorderStyle related implementation.

        /// <summary>
        /// The value of currently used border style.
        /// </summary>
        /// <remarks>
        /// Default border style is <i>Fixed3D</i>.
        /// </remarks>
        /// <seealso cref="BorderStyle"/>
        private BorderStyle borderStyle = BorderStyle.Fixed3D;

        /// <summary>
        /// Gets or sets the value indicating the border style of this 
        /// control.
        /// </summary>
        /// <value>
        /// A value of <i>BorderStyle</i> enumeration. Default value is 
        /// <i>Fixed3D</i>.
        /// </value>
        /// <exception cref="InvalidEnumArgumentException">
        /// This exception is thrown if given border style value is not 
        /// part of the <i>BorderStyle</i> 
        /// enumeration.
        /// </exception>
        /// <seealso cref="borderStyle"/>
        /// <seealso cref="BorderStyleChanged"/>
        /// <seealso cref="OnBorderStyleChanged"/>
        [Category("Appearance")]
        [Description("Indicates the border style of this control.")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(BorderStyle.Fixed3D)]
        public BorderStyle BorderStyle
        {
            get { return this.borderStyle; }
            set
            {
                if (this.borderStyle != value)
                {
                    // Safety check!
                    if (!Enum.IsDefined(typeof(BorderStyle), value))
                    {
                        throw new InvalidEnumArgumentException(
                            "value", (int)value, typeof(BorderStyle));
                    }

                    // Change current value.
                    this.borderStyle = value;

                    // Update native window style.
                    this.UpdateStyles();

                    // Redraw entire control.
                    this.Invalidate();

                    // Raise change event.
                    this.OnBorderStyleChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>BorderStyle</i> has changed.
        /// </summary>
        /// <seealso cref="BorderStyle"/>
        /// <seealso cref="OnBorderStyleChanged"/>
        [Category("Appearance")]
        [Description("Occurs when the value of property BorderStyle has changed.")]
        public event EventHandler<EventArgs> BorderStyleChanged;

        /// <summary>
        /// Raises the <i>BorderStyleChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="BorderStyle"/>
        /// <seealso cref="BorderStyleChanged"/>
        protected virtual void OnBorderStyleChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = BorderStyleChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property BorderStyle related implementation.

        #region Property Image related implementation.

        /// <summary>
        /// The value of currently used picture.
        /// </summary>
        /// <remarks>
        /// Default value is <i>null</i>.
        /// </remarks>
        /// <seealso cref="Image"/>
        private Bitmap image = null;

        /// <summary>
        /// The value of currently used original picture.
        /// </summary>
        /// <remarks>
        /// This value follows the value of member variable <i>image</i>.
        /// </remarks>
        /// <seealso cref="image"/>
        /// <seealso cref="Image"/>
        private Image original = null;

        /// <summary>
        /// Gets or sets the image currently available in this control.
        /// </summary>
        /// <remarks>
        /// The setter creates a copy of given image that is additionally 
        /// adjusted to a proper screen resolution. Furthermore, the pixel 
        /// format of this copy is adapted to speedup a drawing. The getter 
        /// instead always returns the original image!
        /// </remarks>
        /// <value>
        /// The value of currently used picture. Default value is <i>null</i>.
        /// </value>
        /// <seealso cref="image"/>
        /// <seealso cref="ImageChanged"/>
        /// <seealso cref="OnImageChanged"/>
        /// <seealso cref="SetZoomFactor"/>
        /// <seealso cref="UpdateScrollbars"/>
        [Bindable(true)]
        [Localizable(true)]
        [DefaultValue(null)]
        [Category("Appearance")]
        [Description("The image to display in this control.")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public Image Image
        {
            get { return this.original; }
            set
            {
                if (this.image != value)
                {
                    // Release previously used image.
                    if (this.image != null) { this.image.Dispose(); }
                    this.image = null;
                    this.original = null;

                    // Create new image if possible.
                    if (value != null)
                    {
                        using (Graphics graphics = this.CreateGraphics())
                        {
                            Bitmap helper = null;
                            try
                            {
                                // Convert given image into a dedicated bitmap.
                                helper = new Bitmap(value);

                                // Adjust bitmap to screen resolution.
                                helper.SetResolution(graphics.DpiX, graphics.DpiY);

                                // Modify pixel format to accelerate image drawing. 
                                this.image = helper.Clone(
                                    new Rectangle(0, 0, value.Width, value.Height),
                                    PixelFormat.Format32bppPArgb);

                                // Save original reference.
                                this.original = value;
                            }
                            catch
                            {
                                // Release all images in case of a problem.
                                if (this.image != null) { this.image.Dispose(); }
                                this.image = null;
                                this.original = null;
                            }
                            finally
                            {
                                // Free helper image.
                                if (helper != null) { helper.Dispose(); }
                            }
                        }
                    }

                    // Reset current zoom factor to the default.
                    this.SetZoomFactor(this.DefaultZoomFactor);

                    // Reset current scrollbars (auto invalidate).
                    this.UpdateScrollbars(true);

                    // Raise change event.
                    this.OnImageChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>Image</i> has changed.
        /// </summary>
        /// <seealso cref="Image"/>
        /// <seealso cref="OnImageChanged"/>
        [Category("Appearance")]
        [Description("Occurs when the value of property Image has changed.")]
        public event EventHandler<EventArgs> ImageChanged;

        /// <summary>
        /// Raises the <i>ImageChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="Image"/>
        /// <seealso cref="ImageChanged"/>
        protected virtual void OnImageChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = ImageChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property Image related implementation.

        #region Property ErrorText related implementation.

        /// <summary>
        /// The value of currently used error text. 
        /// </summary>
        /// <remarks>
        /// Default value is <i>null</i>.
        /// </remarks>
        /// <seealso cref="ErrorText"/>
        private string errorText = String.Empty;

        /// <summary>
        /// Gets or sets the error text to be used if no image is 
        /// available.
        /// </summary>
        /// <remarks>
        /// Set value of this property to <i>null</i> or empty to 
        /// avoid showing an error text.
        /// </remarks>
        /// <value>
        /// Currently used error text. Default value is <i>empty</i>.
        /// </value>
        /// <seealso cref="errorText"/>
        /// <seealso cref="ErrorTextChanged"/>
        /// <seealso cref="OnErrorTextChanged"/>
        [Category("Appearance")]
        [Description("The error text to be used if no image is available.")]
        [RefreshProperties(RefreshProperties.Repaint)]
        [DefaultValue(typeof(String), "System.String.Empty")]
        public string ErrorText
        {
            get { return String.IsNullOrEmpty(this.errorText) ? String.Empty : this.errorText; }
            set
            {
                if (this.errorText != value)
                {
                    if (String.IsNullOrEmpty(value) && this.errorText != String.Empty)
                    {
                        this.errorText = String.Empty;
                    }
                    else
                    {
                        this.errorText = value;
                    }

                    // Redraw entire control.
                    this.Invalidate();

                    // Raise change event.
                    this.OnErrorTextChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>ErrorText</i> has changed.
        /// </summary>
        /// <seealso cref="ErrorText"/>
        /// <seealso cref="OnErrorTextChanged"/>
        [Category("Appearance")]
        [Description("Occurs when the value of property ErrorText has changed.")]
        public event EventHandler<EventArgs> ErrorTextChanged;

        /// <summary>
        /// Raises the <i>ErrorTextChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="ErrorText"/>
        /// <seealso cref="ErrorTextChanged"/>
        protected virtual void OnErrorTextChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = ErrorTextChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property ErrorText related implementation.

        #region Property ZoomFactor related implementation.

        /// <summary>
        /// The value of currently used zoom factor in percent. 
        /// </summary>
        /// <remarks>
        /// Default value is <i>100%</i>.
        /// </remarks>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ZoomFactorDefaultValue"/>
        private int zoomFactor = ImageBox.ZoomFactorDefaultValue;

        /// <summary>
        /// Gets or sets current zoom factor in percent.
        /// </summary>
        /// <value>
        /// Currently used zoom factor. Default value is <i>100%</i>.
        /// </value>
        /// <exception cref="ArgumentOutOfRangeException">
        /// This exception is thrown if given zoom factor violates 
        /// current limits.
        /// </exception>
        /// <seealso cref="zoomFactor"/>
        /// <seealso cref="ZoomFactorChanged"/>
        /// <seealso cref="OnZoomFactorChanged"/>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="MaximumZoomFactor"/>
        /// <seealso cref="IsValidZoomFactor"/>
        /// <seealso cref="InvokeZoomFactor"/>
        [DefaultValue(ImageBox.ZoomFactorDefaultValue)]
        [Category("Zooming")]
        [Description("The current zoom factor in percent.")]
        [RefreshProperties(RefreshProperties.All)]
        public int ZoomFactor
        {
            get { return this.zoomFactor; }
            set
            {
                if (this.zoomFactor != value)
                {
                    // Safety check!
                    if (!this.IsValidZoomFactor(value))
                    {
                        throw new ArgumentOutOfRangeException(
                            "value", value, String.Format(
                                "Zoom factor must be in range of {0} up to {1}.",
                                this.minimumZoomFactor,
                                this.maximumZoomFactor));
                    }

                    // Set new zoom factor.
                    this.InvokeZoomFactor(value);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>ZoomFactor</i> has changed.
        /// </summary>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="OnZoomFactorChanged"/>
        [Category("Zooming")]
        [Description("Occurs when the value of property ZoomFactor has changed.")]
        public event EventHandler<EventArgs> ZoomFactorChanged;

        /// <summary>
        /// Raises the <i>ZoomFactorChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ZoomFactorChanged"/>
        protected virtual void OnZoomFactorChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = ZoomFactorChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property ZoomFactor related implementation.

        #region Property DefaultZoomFactor related implementation.

        /// <summary>
        /// The internal default zoom factor in percent.
        /// </summary>
        /// <remarks>
        /// The internal default zoom factor is set to a value of 100%.
        /// </remarks>
        private const int ZoomFactorDefaultValue = 100;

        /// <summary>
        /// The value of currently used default zoom factor in percent. 
        /// </summary>
        /// <remarks>
        /// Default value is <i>100%</i>.
        /// </remarks>
        /// <seealso cref="DefaultZoomFactor"/>
        /// <seealso cref="ZoomFactorDefaultValue"/>
        private int defaultZoomFactor = ImageBox.ZoomFactorDefaultValue;

        /// <summary>
        /// Gets or sets current default zoom factor in percent.
        /// </summary>
        /// <value>
        /// Currently used default zoom factor. Default value is 
        /// <i>100%</i>.
        /// </value>
        /// <exception cref="ArgumentOutOfRangeException">
        /// This exception is thrown if given default zoom factor 
        /// violates current limits.
        /// </exception>
        /// <seealso cref="defaultZoomFactor"/>
        /// <seealso cref="DefaultZoomFactorChanged"/>
        /// <seealso cref="OnDefaultZoomFactorChanged"/>
        /// <seealso cref="IsValidZoomFactor"/>
        [DefaultValue(ImageBox.ZoomFactorDefaultValue)]
        [Category("Zooming")]
        [Description("The current default zoom factor in percent.")]
        [RefreshProperties(RefreshProperties.All)]
        public int DefaultZoomFactor
        {
            get { return this.defaultZoomFactor; }
            set
            {
                if (this.defaultZoomFactor != value)
                {
                    // Safety check!
                    if (!this.IsValidZoomFactor(value))
                    {
                        throw new ArgumentOutOfRangeException(
                            "value", value, String.Format(
                                "Default zoom factor must be in range of {0} up to {1}.",
                                this.minimumZoomFactor,
                                this.maximumZoomFactor));
                    }

                    // Set new default zoom factor value.
                    this.defaultZoomFactor = value;

                    // Raise change event.
                    this.OnDefaultZoomFactorChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>DefaultZoomFactor</i> 
        /// has changed.
        /// </summary>
        /// <seealso cref="DefaultZoomFactor"/>
        /// <seealso cref="OnDefaultZoomFactorChanged"/>
        [Category("Zooming")]
        [Description("Occurs when the value of property DefaultZoomFactor has changed.")]
        public event EventHandler<EventArgs> DefaultZoomFactorChanged;

        /// <summary>
        /// Raises the <i>DefaultZoomFactorChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="DefaultZoomFactor"/>
        /// <seealso cref="DefaultZoomFactorChanged"/>
        protected virtual void OnDefaultZoomFactorChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = DefaultZoomFactorChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property DefaultZoomFactor related implementation.

        #region Property MinimumZoomFactor related implementation.

        /// <summary>
        /// The internal minimum zoom factor in percent.
        /// </summary>
        /// <remarks>
        /// The internal minimum zoom factor is set to a value of 10%.
        /// </remarks>
        private const int ZoomFactorMinimumValue = 10;

        /// <summary>
        /// The value of currently used minimum zoom factor in percent. 
        /// </summary>
        /// <remarks>
        /// Default value is <i>10%</i>.
        /// </remarks>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="ZoomFactorMinimumValue"/>
        private int minimumZoomFactor = ImageBox.ZoomFactorMinimumValue;

        /// <summary>
        /// Gets or sets current minimum zoom factor in percent.
        /// </summary>
        /// <remarks>
        /// Changing the value of this property may also modify current 
        /// zoom factor and/or current default zoom factor.
        /// </remarks>
        /// <value>
        /// Currently used minimum zoom factor. Default value is 
        /// <i>10%</i>.
        /// </value>
        /// <exception cref="ArgumentOutOfRangeException">
        /// This exception is thrown if given minimum zoom factor is 
        /// less than zero or greater than current maximum zoom factor.
        /// </exception>
        /// <seealso cref="minimumZoomFactor"/>
        /// <seealso cref="MinimumZoomFactorChanged"/>
        /// <seealso cref="OnMinimumZoomFactorChanged"/>
        /// <seealso cref="MaximumZoomFactor"/>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="DefaultZoomFactor"/>
        [DefaultValue(ImageBox.ZoomFactorMinimumValue)]
        [Category("Zooming")]
        [Description("The current minimum zoom factor in percent.")]
        [RefreshProperties(RefreshProperties.All)]
        public int MinimumZoomFactor
        {
            get { return this.minimumZoomFactor; }
            set
            {
                if (this.minimumZoomFactor != value)
                {
                    // Safety check!
                    if (value <= 0 || value >= this.maximumZoomFactor)
                    {
                        throw new ArgumentOutOfRangeException(
                            "value", value, String.Format(
                                "Minimum zoom factor must be greater than zero and less than the maximum of {0}.",
                                this.maximumZoomFactor));
                    }

                    // Set new zoom factor value.
                    this.minimumZoomFactor = value;

                    // Adjust depending zoom factor if needed!
                    if (this.zoomFactor < value) { this.SetZoomFactor(value); }

                    // Adjust depending default zoom factor if needed!
                    if (this.defaultZoomFactor < value) { this.DefaultZoomFactor = value; }

                    // Raise change event.
                    this.OnMinimumZoomFactorChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>MinimumZoomFactor</i> 
        /// has changed.
        /// </summary>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="OnMinimumZoomFactorChanged"/>
        [Category("Zooming")]
        [Description("Occurs when the value of property MinimumZoomFactor has changed.")]
        public event EventHandler<EventArgs> MinimumZoomFactorChanged;

        /// <summary>
        /// Raises the <i>MinimumZoomFactorChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="MinimumZoomFactorChanged"/>
        protected virtual void OnMinimumZoomFactorChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = MinimumZoomFactorChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property MinimumZoomFactor related implementation.

        #region Property MaximumZoomFactor related implementation.

        /// <summary>
        /// The internal maximum zoom factor in percent.
        /// </summary>
        /// <remarks>
        /// The internal maximum zoom factor is set to a value of 6400%.
        /// </remarks>
        private const int ZoomFactorMaximumValue = 6400;

        /// <summary>
        /// The value of currently used maximum zoom factor in percent. 
        /// </summary>
        /// <remarks>
        /// Default value is <i>6400%</i>.
        /// </remarks>
        /// <seealso cref="MaximumZoomFactor"/>
        /// <seealso cref="ZoomFactorMaximumValue"/>
        private int maximumZoomFactor = ImageBox.ZoomFactorMaximumValue;

        /// <summary>
        /// Gets or sets current maximum zoom factor in percent.
        /// </summary>
        /// <remarks>
        /// Changing the value of this property may also modify current 
        /// zoom factor and/or current default zoom factor.
        /// </remarks>
        /// <value>
        /// Currently used maximum zoom factor. Default value is 
        /// <i>6400%</i>.
        /// </value>
        /// <exception cref="ArgumentOutOfRangeException">
        /// This exception is thrown if given maximum zoom factor is 
        /// less than zero or less than current minimum zoom factor.
        /// </exception>
        /// <seealso cref="maximumZoomFactor"/>
        /// <seealso cref="MaximumZoomFactorChanged"/>
        /// <seealso cref="OnMaximumZoomFactorChanged"/>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="DefaultZoomFactor"/>
        [DefaultValue(ImageBox.ZoomFactorMaximumValue)]
        [Category("Zooming")]
        [Description("The current maximum zoom factor in percent.")]
        [RefreshProperties(RefreshProperties.All)]
        public int MaximumZoomFactor
        {
            get { return this.maximumZoomFactor; }
            set
            {
                if (this.maximumZoomFactor != value)
                {
                    // Safety check!
                    if (value <= 0 || value <= this.minimumZoomFactor)
                    {
                        throw new ArgumentOutOfRangeException(
                            "value", value, String.Format(
                                "Maximum zoom factor must be greater than zero and greater than the minimum of {0}.",
                                this.minimumZoomFactor));
                    }

                    // Set new zoom factor value.
                    this.maximumZoomFactor = value;

                    // Adjust depending zoom factor if needed!
                    if (this.zoomFactor > value) { this.SetZoomFactor(value); }

                    // Adjust depending default zoom factor if needed!
                    if (this.defaultZoomFactor > value) { this.DefaultZoomFactor = value; }

                    // Raise change event.
                    this.OnMaximumZoomFactorChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>MaximumZoomFactor</i> 
        /// has changed.
        /// </summary>
        /// <seealso cref="MaximumZoomFactor"/>
        /// <seealso cref="OnMaximumZoomFactorChanged"/>
        [Category("Zooming")]
        [Description("Occurs when the value of property MaximumZoomFactor has changed.")]
        public event EventHandler<EventArgs> MaximumZoomFactorChanged;

        /// <summary>
        /// Raises the <i>MaximumZoomFactorChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="MaximumZoomFactor"/>
        /// <seealso cref="MaximumZoomFactorChanged"/>
        protected virtual void OnMaximumZoomFactorChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = MaximumZoomFactorChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property MaximumZoomFactor related implementation.

        #region Property Interpolation related implementation.

        /// <summary>
        /// The value of currently used interpolation mode.
        /// </summary>
        /// <remarks>
        ///  Default value is <i>true</i>.
        /// </remarks>
        /// <seealso cref="Interpolation"/>
        private bool interpolation = true;

        /// <summary>
        /// Gets or sets the interpolation mode to draw the image.
        /// </summary>
        /// <remarks>
        /// An interpolation immediatly affects the drawing of an image 
        /// and means how the pixel of an image should be displayed. In 
        /// case of an enabled interpolation each pixel is rendered 
        /// "bilinear". Otherwise each pixel is rendered using the color 
        /// of "nearest neighbor". But this is only relevant if an image 
        /// is zoomed beyond 100 percent. In a lower zooming resolution 
        /// an image is always rendered using the "nearest neighbor" 
        /// color. This behavior is necessary to speedup a painting of 
        /// huge pictures.
        /// </remarks>
        /// <value>
        /// True if interpolation is enabled and false otherwise.
        /// </value>
        /// <seealso cref="interpolation"/>
        /// <seealso cref="InterpolationChanged"/>
        /// <seealso cref="OnInterpolationChanged"/>
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("The interpolation mode to draw the image.")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public bool Interpolation
        {
            get { return this.interpolation; }
            set
            {
                if (this.interpolation != value)
                {
                    this.interpolation = value;

                    // Redraw entire control.
                    this.Invalidate();

                    // Raise change event.
                    this.OnInterpolationChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>Interpolation</i> has changed.
        /// </summary>
        /// <seealso cref="Interpolation"/>
        /// <seealso cref="OnInterpolationChanged"/>
        [Category("Appearance")]
        [Description("Occurs when the value of property Interpolation has changed.")]
        public event EventHandler<EventArgs> InterpolationChanged;

        /// <summary>
        /// Raises the <i>InterpolationChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="Interpolation"/>
        /// <seealso cref="InterpolationChanged"/>
        protected virtual void OnInterpolationChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = InterpolationChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property Interpolation related implementation.

        #region Property SuppressScrollbars related implementation.

        /// <summary>
        /// The value of currently used scrollbar suppressing mode.
        /// </summary>
        /// <remarks>
        ///  Default value is <i>false</i>.
        /// </remarks>
        /// <seealso cref="SuppressScrollbars"/>
        private bool suppressScrollbars = false;

        /// <summary>
        /// Gets or sets the scrollbar suppressing mode.
        /// </summary>
        /// <remarks>
        /// Suppressing scrollbars is primary needed in case of using 
        /// this control as slideshow. This is because loading of huge 
        /// images may take a second or two. Therefore, it could be a 
        /// good idea to hide scrollbars which prevents flickering of 
        /// the scrollbars.
        /// </remarks>
        /// <value>
        /// True if both scrollbars are suppressed and false otherwise.
        /// </value>
        /// <seealso cref="suppressScrollbars"/>
        /// <seealso cref="SuppressScrollbarsChanged"/>
        /// <seealso cref="OnSuppressScrollbarsChanged"/>
        [DefaultValue(true)]
        [Category("Appearance")]
        [Description("Enables or disables scrollbar suppressing mode.")]
        [RefreshProperties(RefreshProperties.Repaint)]
        public bool SuppressScrollbars
        {
            get { return this.suppressScrollbars; }
            set
            {
                if (this.suppressScrollbars != value)
                {
                    this.suppressScrollbars = value;

                    // Apply current scrollbar state.
                    this.UpdateScrollbars(false);

                    // Redraw entire control.
                    this.Invalidate();

                    // Raise change event.
                    this.OnSuppressScrollbarsChanged(EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the value of property <i>Interpolation</i> has changed.
        /// </summary>
        /// <seealso cref="Interpolation"/>
        /// <seealso cref="OnInterpolationChanged"/>
        [Category("Appearance")]
        [Description("Occurs when the value of property SuppressScrollbars has changed.")]
        public event EventHandler<EventArgs> SuppressScrollbarsChanged;

        /// <summary>
        /// Raises the <i>InterpolationChanged</i> event. 
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="Interpolation"/>
        /// <seealso cref="InterpolationChanged"/>
        protected virtual void OnSuppressScrollbarsChanged(EventArgs args)
        {
            EventHandler<EventArgs> handler = SuppressScrollbarsChanged;
            if (handler != null)
            {
                handler(this, args);
            }
        }

        #endregion // Property SuppressScrollbars related implementation.

        #region Inherited property reimplementation.

        /// <summary>
        /// Gets or sets a value indicating whether this control allows 
        /// users to scroll to any controls placed outside of its visible 
        /// boundaries.
        /// </summary>
        /// <remarks>
        /// This inherited property has been overwritten to avoid its usage 
        /// in the Visual Studio designer.
        /// </remarks>
        /// <value>
        /// True if this control enables auto-scrolling; otherwise, false. 
        /// The default value is false.
        /// </value>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override bool AutoScroll
        {
            get { return base.AutoScroll; }
            set { base.AutoScroll = value; }
        }

        /// <summary>
        /// Gets or sets the text associated with this control.
        /// </summary>
        /// <remarks>
        /// This inherited property has been overwritten to avoid its usage 
        /// in the Visual Studio designer.
        /// <para>
        /// Note: Use property <i>ErrorText</i> to associate a text with 
        /// this control.
        /// </para>
        /// </remarks>
        /// <value>
        /// The value of property <i>ErrorText</i>.
        /// </value>
        /// <seealso cref="ErrorText"/>
        [Browsable(false)]
        [DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
        public override string Text
        {
            get { return this.ErrorText; }
            set { this.ErrorText = value; }
        }

        /// <summary>
        /// Gets the required creation parameters when the control handle 
        /// is created.
        /// </summary>
        /// <remarks>
        /// An instance of class <i>CreateParams</i> that contains the 
        /// required creation parameters when the handle to the control 
        /// is created.
        /// </remarks>
        /// <value>
        /// The value of property <i>CreateParams</i>. The constant 
        /// <c>WS_BORDER</c> or <c>WS_EX_STATICEDGE</c> is added 
        /// depending on current border style. 
        /// </value>
        /// <seealso cref="BorderStyle"/>
        protected override CreateParams CreateParams
        {
            get
            {
                // Very smart because client rectangle is also adjusted! 
                // See also: http://support.microsoft.com/kb/316574
                const int WS_BORDER = unchecked((int)0x00800000);
                const int WS_EX_STATICEDGE = unchecked((int)0x00020000);

                CreateParams createParams = base.CreateParams;
                createParams.ExStyle &= (~WS_EX_STATICEDGE);
                createParams.Style &= (~WS_BORDER);

                switch (this.borderStyle)
                {
                    case BorderStyle.Fixed3D:
                        createParams.ExStyle |= WS_EX_STATICEDGE;
                        break;
                    case BorderStyle.FixedSingle:
                        createParams.Style |= WS_BORDER;
                        break;
                }
                return createParams;
            }
        }

        /// <summary>
        /// Gets this control's default size which is 300 by 200 pixels. 
        /// </summary>
        /// <value>
        /// An instance of class <i>Size</i> representing the default 
        /// size of this control.
        /// </value>
        protected override Size DefaultSize
        {
            get { return new Size(300, 200); }
        }

        /// <summary>
        /// Gets this control's default minimum size which is 150 by 50 
        /// pixels. 
        /// </summary>
        /// <value>
        /// An instance of class <i>Size</i> representing the default 
        /// minimum size of this control.
        /// </value>
        protected override Size DefaultMinimumSize
        {
            get { return new Size(150, 50); }
        }

        /// <summary>
        /// Gets this control's additional default padding 
        /// which is 10 for all sides.
        /// </summary>
        /// <value>
        /// An instance of class <i>Padding</i> representing the default 
        /// padding of this control.
        /// </value>
        protected override Padding DefaultPadding
        {
            get { return new Padding(10); }
        }

        #endregion // Inherited property reimplementation.

        #region Rotating related implementation.

        /// <summary>
        /// Occurs when the image has been rotated to the left.
        /// </summary>
        /// <seealso cref="RotateLeft"/>
        [Category("Behavior")]
        [Description("Occurs when the image has been rotated to the left.")]
        public event EventHandler<EventArgs> RotatedLeft;

        /// <summary>
        /// Rotates current content to the left.
        /// </summary>
        /// <remarks>
        /// This member function rotates currently visible image to the left 
        /// which in fact means an anticlockwise rotation. Additionally, current 
        /// image is zoomed in a way that it fits into current client area.
        /// </remarks>
        /// <seealso cref="FitToSize"/>
        /// <seealso cref="RotateRight"/>
        public void RotateLeft()
        {
            if (this.image != null)
            {
                // Rotate and fit image to current display bounds.
                this.image.RotateFlip(RotateFlipType.Rotate90FlipXY);
                this.FitToSize(); // This is what the Windows image view does.

                // Raise rotation event.
                EventHandler<EventArgs> handler = RotatedLeft;
                if (handler != null)
                {
                    handler(this, EventArgs.Empty);
                }
            }
        }

        /// <summary>
        /// Occurs when the image has been rotated to the right.
        /// </summary>
        /// <seealso cref="RotateRight"/>
        [Category("Behavior")]
        [Description("Occurs when the image has been rotated to the right.")]
        public event EventHandler<EventArgs> RotatedRight;

        /// <summary>
        /// Rotates current content to the right.
        /// </summary>
        /// <remarks>
        /// This member function rotates currently visible image to the right 
        /// which in fact means a clockwise rotation. Additionally, current 
        /// image is zoomed in a way that it fits into current client area.
        /// </remarks>
        /// <seealso cref="FitToSize"/>
        /// <seealso cref="RotateLeft"/>
        public void RotateRight()
        {
            if (this.image != null)
            {
                // Rotate and fit image to current display bounds.
                this.image.RotateFlip(RotateFlipType.Rotate270FlipXY);
                this.FitToSize(); // This is what the Windows image view does.

                // Raise rotation event.
                EventHandler<EventArgs> handler = RotatedRight;
                if (handler != null)
                {
                    handler(this, EventArgs.Empty);
                }
            }
        }

        #endregion // Rotating related implementation.

        #region Zooming related implementation.

        /// <summary>
        /// Determines whether given <i>factor</i> is valid or not.
        /// </summary>
        /// <remarks>
        /// A zoom factor must be in the range of minimum up to maximum.
        /// </remarks>
        /// <param name="factor">
        /// The zoom factor to check.
        /// </param>
        /// <returns>
        /// True if given <i>factor</i> is valid and false otherwise.
        /// </returns>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="MaximumZoomFactor"/>
        public bool IsValidZoomFactor(int factor)
        {
            return this.minimumZoomFactor <= factor && factor <= this.maximumZoomFactor;
        }

        /// <summary>
        /// Fits current content onto the client area size.
        /// </summary>
        /// <remarks>
        /// This member function calculates a new zoom factor depending on 
        /// current image size and current client size. This new zoom factor 
        /// is then internally set.
        /// </remarks>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="FitToWidth"/>
        /// <seealso cref="InvokeZoomFactor"/>
        public void FitToSize()
        {
            if (this.image != null)
            {
                float cxImage = (float)this.image.Width;
                float cyImage = (float)this.image.Height;
                float cxClient = (float)this.ClientSize.Width - this.Padding.Horizontal;
                float cyClient = (float)this.ClientSize.Height - this.Padding.Vertical;

                if (this.VScroll) { cxClient += SystemInformation.VerticalScrollBarWidth; }
                if (this.HScroll) { cyClient += SystemInformation.HorizontalScrollBarHeight; }

                // Calculate zoom ratio.
                float xRatio = cxClient / cxImage;
                float yRatio = cyClient / cyImage;

                // Do not rounding but just cut off decimals!
                int factor = (int)(((xRatio < yRatio) ? xRatio : yRatio) * 100f);

                this.InvokeZoomFactor(Math.Max(factor, 1));
            }
        }

        /// <summary>
        /// Fits current content onto the client area width.
        /// </summary>
        /// <remarks>
        /// This member function calculates a new zoom factor depending on 
        /// current image width and current client width. This new zoom 
        /// factor is then internally set.
        /// </remarks>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="FitToSize"/>
        /// <seealso cref="InvokeZoomFactor"/>
        public void FitToWidth()
        {
            if (this.image != null)
            {
                float cxImage = (float)this.image.Width;
                float cxClient = (float)this.ClientSize.Width - this.Padding.Horizontal;

                if (!this.VScroll) { cxClient -= SystemInformation.VerticalScrollBarWidth; }

                // Calculate zoom ratio.
                float xRatio = cxClient / cxImage;

                // Do not rounding but just cut off decimals!
                int factor = (int)(xRatio * 100f);

                this.InvokeZoomFactor(Math.Max(factor, 1));
            }
        }

        /// <summary>
        /// Magnifies current content using default zooming behavior.
        /// </summary>
        /// <remarks>
        /// This function magnifies currently visible content using current 
        /// zoom factor that is multiplied by two. Furthermore, the middle 
        /// of current client area is taken as zooming location.
        /// </remarks>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ZoomIn(PointF)"/>
        /// <seealso cref="ZoomIn(int)"/>
        /// <seealso cref="ZoomIn(PointF,int)"/>
        public void ZoomIn()
        {
            this.Zoom(
                new PointF(
                    (float)this.ClientSize.Width / 2f,
                    (float)this.ClientSize.Height / 2f),
                this.zoomFactor * 2);
        }

        /// <summary>
        /// Magnifies current content using default zooming behavior.
        /// </summary>
        /// <remarks>
        /// This function magnifies currently visible content using current 
        /// zoom factor that is multiplied by two. Furthermore, given point 
        /// is taken as zooming location.
        /// </remarks>
        /// <param name="point">
        /// A point in client coordinates that serves as zooming location.
        /// </param>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ZoomIn()"/>
        /// <seealso cref="ZoomIn(int)"/>
        /// <seealso cref="ZoomIn(PointF,int)"/>
        public void ZoomIn(PointF point)
        {
            this.Zoom(point, this.zoomFactor * 2);
        }

        /// <summary>
        /// Magnifies current content using given zoom factor.
        /// </summary>
        /// <remarks>
        /// This function magnifies currently visible content using given 
        /// zoom factor. Furthermore, the middle of current client area is 
        /// taken as zooming location.
        /// </remarks>
        /// <param name="factor">
        /// The zoom factor to be used.
        /// </param>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomIn()"/>
        /// <seealso cref="ZoomIn(PointF)"/>
        /// <seealso cref="ZoomIn(PointF,int)"/>
        public void ZoomIn(int factor)
        {
            this.Zoom(
                new PointF(
                    (float)this.ClientSize.Width / 2f,
                    (float)this.ClientSize.Height / 2f),
                factor);
        }

        /// <summary>
        /// Magnifies current content at given location using given 
        /// zoom factor.
        /// </summary>
        /// <remarks>
        /// This function magnifies currently visible content using 
        /// given zoom factor. Furthermore, given point is taken as 
        /// zooming location.
        /// </remarks>
        /// <param name="point">
        /// A point in client coordinates that serves as zooming location.
        /// </param>
        /// <param name="factor">
        /// The zoom factor to be used.
        /// </param>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomIn()"/>
        /// <seealso cref="ZoomIn(PointF)"/>
        /// <seealso cref="ZoomIn(int)"/>
        public void ZoomIn(PointF point, int factor)
        {
            this.Zoom(point, factor);
        }

        /// <summary>
        /// Downsizes current content using default zooming behavior.
        /// </summary>
        /// <remarks>
        /// This function downsizes currently visible content using current 
        /// zoom factor that is divided by two. Furthermore, the middle 
        /// of current client area is taken as zooming location.
        /// </remarks>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ZoomOut(PointF)"/>
        /// <seealso cref="ZoomOut(int)"/>
        /// <seealso cref="ZoomOut(PointF,int)"/>
        public void ZoomOut()
        {
            this.Zoom(
                new PointF(
                    (float)this.ClientSize.Width / 2f,
                    (float)this.ClientSize.Height / 2f),
                this.zoomFactor / 2);
        }

        /// <summary>
        /// Downsizes current content using default zooming behavior.
        /// </summary>
        /// <remarks>
        /// This function downsizes currently visible content using current 
        /// zoom factor that is divided by two. Furthermore, given point is 
        /// taken as zooming location.
        /// </remarks>
        /// <param name="point">
        /// A point in client coordinates that serves as zooming location.
        /// </param>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ZoomOut()"/>
        /// <seealso cref="ZoomOut(int)"/>
        /// <seealso cref="ZoomOut(PointF,int)"/>
        public void ZoomOut(PointF point)
        {
            this.Zoom(point, this.zoomFactor / 2);
        }

        /// <summary>
        /// Downsizes current content using given zoom factor.
        /// </summary>
        /// <remarks>
        /// This function downsizes currently visible content using given 
        /// zoom factor. Furthermore, the middle of current client area 
        /// is taken as zooming location.
        /// </remarks>
        /// <param name="factor">
        /// The zoom factor to be used.
        /// </param>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomOut()"/>
        /// <seealso cref="ZoomOut(PointF)"/>
        /// <seealso cref="ZoomOut(PointF,int)"/>
        public void ZoomOut(int factor)
        {
            this.Zoom(
                new PointF(
                    (float)this.ClientSize.Width / 2f,
                    (float)this.ClientSize.Height / 2f),
                factor);
        }

        /// <summary>
        /// Downsizes current content at given location using given 
        /// zoom factor.
        /// </summary>
        /// <remarks>
        /// This function downsizes currently visible content using 
        /// given zoom factor. Furthermore, given point is taken as 
        /// zooming location.
        /// </remarks>
        /// <param name="point">
        /// A point in client coordinates that serves as zooming location.
        /// </param>
        /// <param name="factor">
        /// The zoom factor to be used.
        /// </param>
        /// <seealso cref="Zoom"/>
        /// <seealso cref="ZoomOut()"/>
        /// <seealso cref="ZoomOut(PointF)"/>
        /// <seealso cref="ZoomOut(int)"/>
        public void ZoomOut(PointF point, int factor)
        {
            this.Zoom(point, factor);
        }

        /// <summary>
        /// Changes current zoom factor.
        /// </summary>
        /// <remarks>
        /// This method performs a safety check before 
        /// property <i>ZoomFactor</i> is invoked with 
        /// given factor.
        /// </remarks>
        /// <param name="factor">
        /// The new zoom factor to be used.
        /// </param>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="IsValidZoomFactor"/>
        protected void SetZoomFactor(int factor)
        {
            // Safety check to avoid an exception.
            if (this.IsValidZoomFactor(factor))
            {
                this.ZoomFactor = factor;
            }
        }

        /// <summary>
        /// Changes current zoom factor.
        /// </summary>
        /// <remarks>
        /// This method checks only whether given factor is greater than 
        /// zero. And if so then value of <i>zoomFactor</i> is changed. 
        /// Furthermore, the scrollbars are updated as well as the event 
        /// <i>ZoomFactorChanged</i> is raised.
        /// </remarks>
        /// <param name="factor">
        /// The new zoom factor to be used.
        /// </param>
        /// <seealso cref="ZoomFactor"/>
        private void InvokeZoomFactor(int factor)
        {
            if (factor > 0)
            {
                // Change current zoom factor value.
                this.zoomFactor = factor;

                // Update current scrollbars (auto invalidate).
                this.UpdateScrollbars(false);

                // Raise change event.
                this.OnZoomFactorChanged(EventArgs.Empty);
            }
        }

        /// <summary>
        /// Magnifies or downsizes current content at given location 
        /// using given zoom factor.
        /// </summary>
        /// <remarks>
        /// This member function magnifies or downsizes current content 
        /// at given location using given zoom factor. Furthermore, this 
        /// zooming operation depends on current scaled image size as well 
        /// as on current zoom factor. Both values are taken to calculate 
        /// a zooming ratio which then is used to calculate the new scroll 
        /// positions. Additionally, note that the given zoom factor is 
        /// automatically adjusted if it exceeds the current limits.
        /// </remarks>
        /// <param name="point">
        /// A point in client coordinates that serves as zooming location.
        /// </param>
        /// <param name="factor">
        /// The zoom factor to be used.
        /// </param>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="MinimumZoomFactor"/>
        /// <seealso cref="MaximumZoomFactor"/>
        /// <seealso cref="ScaledImageOffset"/>
        /// <seealso cref="SetZoomFactor"/>
        /// <seealso cref="BeginUpdate"/>
        /// <seealso cref="EndUpdate"/>
        private void Zoom(PointF point, int factor)
        {
            // Get current zoom factor locally.
            int current = this.zoomFactor;

            // Adjust local zoom factor to maximum if necessary. 
            if (current > this.maximumZoomFactor)
            {
                // Might happen because member functions 
                // FitToXXX() are able to change current 
                // zoom factor beyond allowed maximum!
                current = this.maximumZoomFactor;
            }

            // Adjust local zoom factor to minimum if necessary. 
            if (current < this.minimumZoomFactor)
            {
                // Might happen because member functions 
                // FitToXXX() are able to change current 
                // zoom factor under allowed minimum!
                current = this.minimumZoomFactor;
            }

            // Limit given zoom factor to maximum if necessary. 
            if (factor > this.maximumZoomFactor)
            {
                factor = this.maximumZoomFactor;
            }

            // Limit given zoom factor to minimum if necessary. 
            if (factor < this.minimumZoomFactor)
            {
                factor = this.minimumZoomFactor;
            }

            // Do not execute if real zoom is equal.
            if (factor == this.zoomFactor)
            {
                return;
            }

            this.BeginUpdate();
            try
            {
                // Calculate new zooming ratio.
                float ratio = (float)factor / (float)current;

                // New scroll position depends on 
                // current scaled image offset!
                PointF offset = this.ScaledImageOffset;

                // Calculate new scrolling offsets.
                float xScroll = (point.X - offset.X) * ratio - point.X + this.Padding.Left;
                float yScroll = (point.Y - offset.Y) * ratio - point.Y + this.Padding.Top;

                // Save current client bounds.
                RectangleF oldClient = this.ClientRectangle;

                // Set new zoom factor.
                this.SetZoomFactor(factor);

                // Get new client bounds.
                RectangleF newClient = this.ClientRectangle;

                // Correct X scrolling value in case 
                // of a client width reduction.
                if (oldClient.Width > newClient.Width)
                {
                    xScroll += (oldClient.Width - newClient.Width) / ratio;
                }

                // Correct Y scrolling value in case 
                // of a client height reduction.
                if (oldClient.Height > newClient.Height)
                {
                    yScroll += (oldClient.Height - newClient.Height) / ratio;
                }

                // Set new scrolling position.
                this.AutoScrollPosition = new Point(
                    (int)Math.Round(xScroll, 0),
                    (int)Math.Round(yScroll, 0));
            }
            finally
            {
                this.EndUpdate();
            }
        }

        #endregion // Zooming related implementation.

        #region Drawing related member function implementation.

        /// <summary>
        /// Paints the background of the control.
        /// </summary>
        /// <remarks>
        /// Painting the disabled background is only handled by this 
        /// member function. Otherwise the inherited method is called.
        /// </remarks>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        protected override void OnPaintBackground(PaintEventArgs args)
        {
            // Let the parent do the job if control enabled.
            if (this.Enabled)
            {
                base.OnPaintBackground(args);
            }
            // Otherwise, reflect that control is disabled.
            else
            {
                // REMARK: Read-only color doesn't make sense because it's a viewer.
                args.Graphics.Clear(SystemColors.Control);
            }
        }

        /// <summary>
        /// Paints the foreground of the control.
        /// </summary>
        /// <remarks>
        /// This method draws current image at current location using 
        /// current zoom factor. If an image is not available at the 
        /// moment then an error text is drawn instead.
        /// </remarks>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="RenderError"/>
        /// <seealso cref="RenderImage"/>
        /// <seealso cref="Interpolation"/>
        /// <seealso cref="ZoomFactor"/>
        protected override void OnPaint(PaintEventArgs args)
        {
            if (this.image == null)
            {
                this.RenderError(args.Graphics);
            }
            else
            {
                // Adjust drawing mode accordingly.
                if (this.zoomFactor < 100)
                {
                    args.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor;
                }
                else
                {
                    if (this.interpolation)
                    {
                        args.Graphics.InterpolationMode = InterpolationMode.Bilinear;
                    }
                    else
                    {
                        args.Graphics.InterpolationMode = InterpolationMode.NearestNeighbor;
                    }
                }

                Matrix transform = args.Graphics.Transform;
                this.RenderImage(args.Graphics);
                args.Graphics.Transform = transform;
            }

            // Call as last to ensure Paint event.
            base.OnPaint(args);
        }

        /// <summary>
        /// Draws an error text if no image is currently available.
        /// </summary>
        /// <remarks>
        /// If the control is currently disabled then the error 
        /// text is drawn using a gray text color.
        /// </remarks>
        /// <param name="graphics">
        /// The graphics context to draw the error text on.
        /// </param>
        /// <seealso cref="ErrorText"/>
        private void RenderError(Graphics graphics)
        {
            if (!String.IsNullOrEmpty(this.errorText))
            {
                // REMARK: Read-only color doesn't make sense because it's a viewer.
                Color foreground = this.Enabled ? this.ForeColor : SystemColors.GrayText;
                using (Brush brush = new SolidBrush(foreground))
                {
                    StringFormat format = new StringFormat();
                    format.LineAlignment = StringAlignment.Center;
                    format.Alignment = StringAlignment.Center;

                    graphics.DrawString(this.errorText, this.Font,
                        brush, this.ClientRectangle, format);
                }
            }
        }

        /// <summary>
        /// Draws currently available image.
        /// </summary>
        /// <remarks>
        /// If the control is currently disabled then the image 
        /// is drawn using a gray foreground.
        /// </remarks>
        /// <param name="graphics">
        /// The graphics context to draw the image on.
        /// </param>
        /// <seealso cref="Image"/>
        private void RenderImage(Graphics graphics)
        {
            float factor = (float)this.zoomFactor / 100f;
            Matrix matrix = new Matrix();
            matrix.Scale(factor, factor, MatrixOrder.Append);
            graphics.Transform = matrix;

            // Determine horizontal and vertical image position.
            PointF offset = this.ScaledImageOffset;

            // Now convert into zooming position!
            float xOffset = offset.X / factor;
            float yOffset = offset.Y / factor;

            // Draw current image.
            if (this.Enabled)
            {
                graphics.DrawImage(this.image, xOffset, yOffset);
            }
            else
            {
                ControlPaint.DrawImageDisabled(
                    graphics, this.image,
                    (int)Math.Round(xOffset, 0),
                    (int)Math.Round(yOffset, 0),
                    SystemColors.Control);
            }
        }

        /// <summary>
        /// Suppresses the redrawing of this control.
        /// </summary>
        /// <remarks>
        /// This member function switches off any kind of redrawing until 
        /// member function <see cref="EndUpdate()"/> is called.
        /// </remarks>
        /// <seealso cref="EndUpdate"/>
        /// <seealso cref="SendMessage"/>
        /// <seealso cref="WM_SETREDRAW"/>
        public void BeginUpdate()
        {
            // Switch redraw off.
            ImageBox.SendMessage(this.Handle, WM_SETREDRAW, (IntPtr)0, IntPtr.Zero);
        }

        /// <summary>
        /// Resumes painting after redrawing has been suppressed.
        /// </summary>
        /// <remarks>
        /// This member function switches on the redrawing of this control if it 
        /// was suppressed by calling member function <see cref="BeginUpdate"/>.
        /// </remarks>
        /// <seealso cref="BeginUpdate"/>
        /// <seealso cref="SendMessage"/>
        /// <seealso cref="WM_SETREDRAW"/>
        public void EndUpdate()
        {
            // Switch redraw on and invalidate entire control.
            ImageBox.SendMessage(this.Handle, WM_SETREDRAW, (IntPtr)1, IntPtr.Zero);
            this.Invalidate();
        }

        #endregion // Drawing related member function implementation.

        #region Keyboard related member function implementation.

        /// <summary>
        /// Indicates whether no additional key is pressed.
        /// </summary>
        /// <value>
        /// True if no additional key is pressed and false otherwise.
        /// </value>
        private bool IsNoneKey
        {
            get
            {
                return (Control.ModifierKeys == Keys.None);
            }
        }

        /// <summary>
        /// Indicates whether the <b>SHIFT</b> key is pressed.
        /// </summary>
        /// <value>
        /// True if the <b>SHIFT</b> key is pressed and false otherwise.
        /// </value>
        private bool IsShiftKey
        {
            get
            {
                return ((Control.ModifierKeys & Keys.Shift) == Keys.Shift);
            }
        }

        /// <summary>
        /// Indicates whether the <b>CTRL</b> key is pressed.
        /// </summary>
        /// <value>
        /// True if the <b>CTRL</b> key is pressed and false otherwise.
        /// </value>
        private bool IsControlKey
        {
            get
            {
                return ((Control.ModifierKeys & Keys.Control) == Keys.Control);
            }
        }

        /// <summary>
        /// Preprocesses keyboard or input messages within the message loop before they 
        /// are dispatched.
        /// </summary>
        /// <param name="message">
        /// An instance of class Message, passed by reference, that represents the message 
        /// to process.
        /// </param>
        /// <returns>
        /// True if the message was processed and false otherwise.
        /// </returns>
        /// <seealso cref="ProcessKeyDown"/>
        [SecurityPermission(SecurityAction.LinkDemand, UnmanagedCode = true),
        SecurityPermission(SecurityAction.InheritanceDemand, UnmanagedCode = true)]
        public override bool PreProcessMessage(ref Message message)
        {
            const int WM_KEYDOWN = 0x0100;

            if (message.Msg == WM_KEYDOWN && this.ProcessKeyDown(ref message))
            {
                return true;
            }
            else
            {
                return base.PreProcessMessage(ref message);
            }
        }

        /// <summary>
        /// Preprocesses the <i>key down</i> keyboard message.
        /// </summary>
        /// <remarks>
        /// See list below which keyboard keys and shortcuts are currently 
        /// processed by this method.
        /// <para>
        /// <list type="table">
        /// <listheader>
        ///     <term>
        ///     Keys
        ///     </term>
        ///     <description>
        ///     Description
        ///     </description>
        /// </listheader>
        /// <item>
        ///     <term>
        ///     HOME
        ///     </term>
        ///     <description>
        ///     Processes the <b>HOME</b> key and all related key combinations. See member 
        ///     function <see cref="ScrollHome()"/> for more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     END
        ///     </term>
        ///     <description>
        ///     Processes the <b>END</b> key and all related key combinations. See member 
        ///     function <see cref="ScrollEnd()"/> for more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     UP
        ///     </term>
        ///     <description>
        ///     Processes the <b>UP</b> key and all related key combinations. See member 
        ///     functions <see cref="ScrollUp()"/> and <see cref="ScrollPageUp()"/> for 
        ///     more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     DOWN
        ///     </term>
        ///     <description>
        ///     Processes the <b>DOWN</b> key and all related key combinations. See member 
        ///     functions <see cref="ScrollDown()"/> and <see cref="ScrollPageDown()"/> for 
        ///     more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     LEFT
        ///     </term>
        ///     <description>
        ///     Processes the <b>LEFT</b> key and all related key combinations. See member 
        ///     functions <see cref="ScrollLeft()"/> and <see cref="ScrollPageLeft()"/> for 
        ///     more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     RIGHT
        ///     </term>
        ///     <description>
        ///     Processes the <b>RIGHT</b> key and all related key combinations. See member 
        ///     functions <see cref="ScrollRight()"/> and <see cref="ScrollPageRight()"/> for 
        ///     more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     PAGE UP
        ///     </term>
        ///     <description>
        ///     Processes the <b>PAGE UP</b> key and all related key combinations. See member 
        ///     functions <see cref="ScrollPageUp()"/> and <see cref="ScrollPageLeft()"/> for 
        ///     more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     PAGE DOWN
        ///     </term>
        ///     <description>
        ///     Processes the <b>PAGE DOWN</b> key and all related key combinations. See member 
        ///     functions <see cref="ScrollPageDown()"/> and <see cref="ScrollPageRight()"/> for 
        ///     more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     CTRL+PLUS
        ///     </term>
        ///     <description>
        ///     Processes the <b>CTRL+PLUS</b> keyboard shortcut. See member 
        ///     function <see cref="ZoomIn()"/> for more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     CTRL+MINUS
        ///     </term>
        ///     <description>
        ///     Processes the <b>CTRL+MINUS</b> keyboard shortcut. See member 
        ///     function <see cref="ZoomOut()"/> for more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     CTRL+ZERO
        ///     </term>
        ///     <description>
        ///     Processes the <b>CTRL+ZERO</b> keyboard shortcut. See member 
        ///     functions <see cref="SetZoomFactor">SetZoomFactor()</see> and 
        ///     <see cref="DefaultZoomFactor"/> for more details.
        ///     </description>
        /// </item>
        /// <item>
        ///     <term>
        ///     TAB
        ///     </term>
        ///     <description>
        ///     Processes the <b>TAB</b> key and puts the keyboard input focus either onto 
        ///     the next control or onto the previous control if the <b>SHIFT</b> is pressed.
        ///     </description>
        /// </item>
        /// </list>
        /// </para>
        /// </remarks>
        /// <param name="message">
        /// An instance of class Message, passed by reference, that represents the message 
        /// to process.
        /// </param>
        /// <returns>
        /// True if the message was processed and false otherwise.
        /// </returns>
        /// <seealso cref="PreProcessMessage"/>
        private bool ProcessKeyDown(ref Message message)
        {
            Keys keys = (Keys)message.WParam.ToInt32();

            if (keys == Keys.Home)
            {
                this.ScrollHome();
                return true;
            }
            else if (keys == Keys.End)
            {
                this.ScrollEnd();
                return true;
            }
            else if (keys == Keys.Up)
            {
                if (this.IsControlKey)
                {
                    this.ScrollPageUp();
                }
                else
                {
                    this.ScrollUp();
                }
                return true;
            }
            else if (keys == Keys.Down)
            {
                if (this.IsControlKey)
                {
                    this.ScrollPageDown();
                }
                else
                {
                    this.ScrollDown();
                }
                return true;
            }
            else if (keys == Keys.Left)
            {
                if (this.IsControlKey)
                {
                    this.ScrollPageLeft();
                }
                else
                {
                    this.ScrollLeft();
                }
                return true;
            }
            else if (keys == Keys.Right)
            {
                if (this.IsControlKey)
                {
                    this.ScrollPageRight();
                }
                else
                {
                    this.ScrollRight();
                }
                return true;
            }
            else if (keys == Keys.PageUp)
            {
                if (this.IsControlKey)
                {
                    this.ScrollPageLeft();
                }
                else
                {
                    this.ScrollPageUp();
                }
                return true;
            }
            else if (keys == Keys.PageDown)
            {
                if (this.IsControlKey)
                {
                    this.ScrollPageRight();
                }
                else
                {
                    this.ScrollPageDown();
                }
                return true;
            }
            else if ((keys == Keys.Add || keys == Keys.Oemplus) && this.IsControlKey)
            {
                this.ZoomIn();
                return true;
            }
            else if ((keys == Keys.Subtract || keys == Keys.OemMinus) && this.IsControlKey)
            {
                this.ZoomOut();
                return true;
            }
            else if ((keys == Keys.D0 || keys == Keys.NumPad0) && this.IsControlKey)
            {
                this.SetZoomFactor(this.DefaultZoomFactor);
                return true;
            }
            else if (keys == Keys.Tab)
            {
                // Try find next selectable control and jump to.
                Control parent = this.Parent;
                while (parent != null)
                {
                    if (parent.CanSelect)
                    {
                        return parent.SelectNextControl(this, !this.IsShiftKey, true, true, true);
                    }
                    parent = parent.Parent;
                }
                return false;
            }
            else
            {
                return false;
            }
        }

        #endregion // Keyboard related member function implementation.

        #region Mouse event handler reimplementation.

        /// <summary>
        /// Sets the input focus to the control and raises the 
        /// <i>MouseDown</i> event.
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        protected override void OnMouseDown(MouseEventArgs args)
        {
            // Activate control if not yet activated.
            if (!this.Focused) { this.Focus(); }

            if (args.Button == MouseButtons.Left)
            {
                #region Mouse dragging handling.

                // Save current mouse down location that indicates current grab offset.
                this.dragOffset = new Point(args.Location.X, args.Location.Y);

                #endregion // Mouse dragging handling.
            }

            // Call as last to ensure MouseDown event.
            base.OnMouseDown(args);
        }

        /// <summary>
        /// Handles image dragging if needed and raises the 
        /// <i>MouseMove</i> event.
        /// </summary>
        /// <remarks>
        /// If left mouse button is pressed then current 
        /// image is scrolled.
        /// </remarks>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        protected override void OnMouseMove(MouseEventArgs args)
        {
            if (args.Button == MouseButtons.Left)
            {
                #region Mouse dragging handling.

                if (this.HScroll)
                {
                    if (this.dragOffset.X < args.Location.X)
                    {
                        this.ScrollLeft(args.Location.X - this.dragOffset.X);
                    }
                    else
                    {
                        this.ScrollRight(this.dragOffset.X - args.Location.X);
                    }
                }

                if (this.VScroll)
                {
                    if (this.dragOffset.Y < args.Location.Y)
                    {
                        this.ScrollUp(args.Location.Y - this.dragOffset.Y);
                    }
                    else
                    {
                        this.ScrollDown(this.dragOffset.Y - args.Location.Y);
                    }
                }

                // Save new mouse location that indicates new grab offset.
                this.dragOffset = new Point(args.Location.X, args.Location.Y);

                #endregion // Mouse dragging handling.
            }

            // Call as last to ensure MouseDown event.
            base.OnMouseMove(args);
        }

        /// <summary>
        /// Releases image dragging if needed and raises the 
        /// <i>MouseUp</i> event.
        /// </summary>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        protected override void OnMouseUp(MouseEventArgs args)
        {
            if (args.Button == MouseButtons.Left)
            {
                #region Mouse dragging handling.

                // Release current mouse grab offset.
                this.dragOffset = Point.Empty;

                #endregion // Mouse dragging handling.
            }

            // Call as last to ensure MouseDown event.
            base.OnMouseUp(args);
        }

        /// <summary>
        /// Scrolls or zooms the visible content depending on additional 
        /// modifier keys and raises the <i>MouseWheel</i> event.
        /// </summary>
        /// <remarks>
        /// If no additional modifier key is pressed then current content 
        /// is scrolled up or down. If modifier key <b>SHIFT</b> is pressed 
        /// then current content is scrolled left or right. If modifier key 
        /// <b>CTRL</b> is pressed instead then current content is zoomed 
        /// in or out.
        /// </remarks>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        protected override void OnMouseWheel(MouseEventArgs args)
        {
            if (args.Button == MouseButtons.None)
            {
                // Scrolling up/down on mouse wheel.
                if (this.IsNoneKey)
                {
                    if (args.Delta > 0)
                    {
                        this.ScrollUp();
                    }
                    else if (args.Delta < 0)
                    {
                        this.ScrollDown();
                    }
                }
                // Scrolling left/right on mouse wheel.
                else if (this.IsShiftKey)
                {
                    if (args.Delta > 0)
                    {
                        this.ScrollLeft();
                    }
                    else if (args.Delta < 0)
                    {
                        this.ScrollRight();
                    }
                }
                // Zooming in/out on mouse wheel.
                else if (this.IsControlKey)
                {
                    if (args.Delta > 0)
                    {
                        this.ZoomIn(args.Location);
                    }
                    else if (args.Delta < 0)
                    {
                        this.ZoomOut(args.Location);
                    }
                }
            }

            // NOTE: Suppress scrolling in the base class but fire MouseWheel event.
            // Inherit member function OnMouseWheel() cause some trouble when trying to scroll 
            // horizontal using SHIFT key. In this case, that member function scrolls vertical 
            // too! Therefore, it was necessary to "overwrite" the MouseWheel event declaration 
            // and to fire this event explicitly!
            if (MouseWheel != null) { MouseWheel(this, args); }
        }

        /// <summary>
        /// Occurs when the mouse wheel moves while the control has focus.
        /// </summary>
        /// <remarks>
        /// This code was just copied from class <i>Control</i>. But it is 
        /// really needed because it prevents class <i>ScrollableControl</i> 
        /// from scrolling.
        /// </remarks>
        [Browsable(false)]
        [EditorBrowsable(EditorBrowsableState.Advanced)]
        public new event MouseEventHandler MouseWheel;

        #endregion // Mouse event handler reimplementation.

        #region Scrolling related member function implementation.

        /// <summary>
        /// Scrolls the visible content to one of the possible 
        /// HOME positions. 
        /// </summary>
        /// <remarks>
        /// If no additional modifier key is pressed then the new 
        /// HOME position is set to the left. If the <b>SHIFT</b> 
        /// key is additionally pressed then the new HOME position 
        /// is set to the top. If the <b>CTRL</b> key is pressed 
        /// instead then the new HOME position is set to the upper 
        /// left corner.
        /// </remarks>
        /// <seealso cref="ScrollEnd"/>
        protected void ScrollHome()
        {
            int x = Math.Abs(this.AutoScrollPosition.X);
            int y = Math.Abs(this.AutoScrollPosition.Y);

            // Jump to first X position.
            // Current Y position remains!
            if (!this.IsShiftKey && !this.IsControlKey)
            {
                x = this.HorizontalScroll.Minimum;
            }
            // Jump to first Y position.
            // Current X position remains!
            else if (this.IsShiftKey && !this.IsControlKey)
            {
                y = this.VerticalScroll.Minimum;
            }
            // Jump to first X position.
            // Jump to first Y position.
            else if (!this.IsShiftKey && this.IsControlKey)
            {
                x = this.HorizontalScroll.Minimum;
                y = this.VerticalScroll.Minimum;
            }
            // Suppress any other combination!
            else
            {
                return;
            }

            // Limitation or a range check is not necessary 
            // because property AutoScrollPosition handles 
            // range violations accordingly.

            this.AutoScrollPosition = new Point(x, y);
        }

        /// <summary>
        /// Scrolls the visible content to one of the possible 
        /// END positions. 
        /// </summary>
        /// <remarks>
        /// If no additional modifier key is pressed then the new 
        /// END position is set to the left. If the <b>SHIFT</b> 
        /// key is additionally pressed then the new END position 
        /// is set to the top. If the <b>CTRL</b> key is pressed 
        /// instead then the new END position is set to the upper 
        /// left corner.
        /// </remarks>
        /// <seealso cref="ScrollHome"/>
        protected void ScrollEnd()
        {
            int x = Math.Abs(this.AutoScrollPosition.X);
            int y = Math.Abs(this.AutoScrollPosition.Y);

            // Jump to first X position.
            // Current Y position remains!
            if (!this.IsShiftKey && !this.IsControlKey)
            {
                x = this.HorizontalScroll.Maximum;
            }
            // Jump to first Y position.
            // Current X position remains!
            else if (this.IsShiftKey && !this.IsControlKey)
            {
                y = this.VerticalScroll.Maximum;
            }
            // Jump to first X position.
            // Jump to first Y position.
            else if (!this.IsShiftKey && this.IsControlKey)
            {
                x = this.HorizontalScroll.Maximum;
                y = this.VerticalScroll.Maximum;
            }
            // Suppress any other combination!
            else
            {
                return;
            }

            // Limitation or a range check is not necessary 
            // because property AutoScrollPosition handles 
            // range violations accordingly.

            this.AutoScrollPosition = new Point(x, y);
        }

        /// <summary>
        /// Scrolls current content upward.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>SmallChange</i> of 
        /// the vertical scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollUp(int)"/>
        protected void ScrollUp()
        {
            this.ScrollUp(this.VerticalScroll.SmallChange);
        }

        /// <summary>
        /// Scrolls current content upward using given amount.
        /// </summary>
        /// <param name="delta">
        /// The relative amount to scroll current content upward.
        /// </param>
        /// <seealso cref="ScrollUp()"/>
        protected void ScrollUp(int delta)
        {
            int x = Math.Abs(this.AutoScrollPosition.X);
            int y = Math.Abs(this.AutoScrollPosition.Y);

            // Limitation or a range check is not necessary 
            // because property AutoScrollPosition handles 
            // range violations accordingly.
            y -= delta;

            this.AutoScrollPosition = new Point(x, y);
        }

        /// <summary>
        /// Scrolls current content downward.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>SmallChange</i> of 
        /// the vertical scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollDown(int)"/>
        protected void ScrollDown()
        {
            this.ScrollDown(this.VerticalScroll.SmallChange);
        }

        /// <summary>
        /// Scrolls current content downward using given amount.
        /// </summary>
        /// <param name="delta">
        /// The relative amount to scroll current content downward.
        /// </param>
        /// <seealso cref="ScrollDown()"/>
        protected void ScrollDown(int delta)
        {
            int x = Math.Abs(this.AutoScrollPosition.X);
            int y = Math.Abs(this.AutoScrollPosition.Y);

            // Limitation or a range check is not necessary 
            // because property AutoScrollPosition handles 
            // range violations accordingly.
            y += delta;

            this.AutoScrollPosition = new Point(x, y);
        }

        /// <summary>
        /// Scrolls current content to the left.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>SmallChange</i> of 
        /// the horizontal scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollLeft(int)"/>
        protected void ScrollLeft()
        {
            this.ScrollLeft(this.VerticalScroll.SmallChange);
        }

        /// <summary>
        /// Scrolls current content to the left using given amount.
        /// </summary>
        /// <param name="delta">
        /// The relative amount to scroll current content to the left.
        /// </param>
        /// <seealso cref="ScrollLeft()"/>
        protected void ScrollLeft(int delta)
        {
            int x = Math.Abs(this.AutoScrollPosition.X);
            int y = Math.Abs(this.AutoScrollPosition.Y);

            // Limitation or a range check is not necessary 
            // because property AutoScrollPosition handles 
            // range violations accordingly.
            x -= delta;

            this.AutoScrollPosition = new Point(x, y);
        }

        /// <summary>
        /// Scrolls current content to the right.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>SmallChange</i> of 
        /// the horizontal scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollRight(int)"/>
        protected void ScrollRight()
        {
            this.ScrollRight(this.VerticalScroll.SmallChange);
        }

        /// <summary>
        /// Scrolls current content to the right using given amount.
        /// </summary>
        /// <param name="delta">
        /// The relative amount to scroll current content to the right.
        /// </param>
        /// <seealso cref="ScrollRight()"/>
        protected void ScrollRight(int delta)
        {
            int x = Math.Abs(this.AutoScrollPosition.X);
            int y = Math.Abs(this.AutoScrollPosition.Y);

            // Limitation or a range check is not necessary 
            // because property AutoScrollPosition handles 
            // range violations accordingly.
            x += delta;

            this.AutoScrollPosition = new Point(x, y);
        }

        /// <summary>
        /// Scrolls the visible content one page up.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>LargeChange</i> of 
        /// the vertical scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollPageUp(int)"/>
        protected void ScrollPageUp()
        {
            this.ScrollUp(this.VerticalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content upward using given pages.
        /// </summary>
        /// <remarks>
        /// Given number of pages is multiplied by current value of 
        /// property <i>LargeChange</i> of the vertical scrollbar and 
        /// is then used as scroll amount.
        /// </remarks>
        /// <param name="pages">
        /// The number of pages to scroll upward.
        /// </param>
        /// <seealso cref="ScrollPageUp()"/>
        /// <seealso cref="ScrollUp(int)"/>
        protected void ScrollPageUp(int pages)
        {
            this.ScrollUp(pages * this.VerticalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content one page down.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>LargeChange</i> of 
        /// the vertical scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollPageDown(int)"/>
        protected void ScrollPageDown()
        {
            this.ScrollDown(this.VerticalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content downward using given pages.
        /// </summary>
        /// <remarks>
        /// Given number of pages is multiplied by current value of 
        /// property <i>LargeChange</i> of the vertical scrollbar and 
        /// is then used as scroll amount.
        /// </remarks>
        /// <param name="pages">
        /// The number of pages to scroll downward.
        /// </param>
        /// <seealso cref="ScrollPageDown()"/>
        /// <seealso cref="ScrollDown(int)"/>
        protected void ScrollPageDown(int pages)
        {
            this.ScrollDown(pages * this.VerticalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content one page to the left.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>LargeChange</i> of 
        /// the horizontal scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollPageLeft(int)"/>
        protected void ScrollPageLeft()
        {
            this.ScrollLeft(this.HorizontalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content to the left using given pages.
        /// </summary>
        /// <remarks>
        /// Given number of pages is multiplied by current value of 
        /// property <i>LargeChange</i> of the horizontal scrollbar 
        /// and is then used as scroll amount.
        /// </remarks>
        /// <param name="pages">
        /// The number of pages to scroll to the left.
        /// </param>
        /// <seealso cref="ScrollPageLeft()"/>
        /// <seealso cref="ScrollLeft(int)"/>
        protected void ScrollPageLeft(int pages)
        {
            this.ScrollLeft(pages * this.HorizontalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content one page to the right.
        /// </summary>
        /// <remarks>
        /// Current value of property <i>LargeChange</i> of 
        /// the horizontal scrollbar is used as scroll amount.
        /// </remarks>
        /// <seealso cref="ScrollPageRight(int)"/>
        protected void ScrollPageRight()
        {
            this.ScrollRight(this.HorizontalScroll.LargeChange);
        }

        /// <summary>
        /// Scrolls the visible content to the right using given pages.
        /// </summary>
        /// <remarks>
        /// Given number of pages is multiplied by current value of 
        /// property <i>LargeChange</i> of the horizontal scrollbar 
        /// and is then used as scroll amount.
        /// </remarks>
        /// <param name="pages">
        /// The number of pages to scroll to the right.
        /// </param>
        /// <seealso cref="ScrollPageLeft()"/>
        /// <seealso cref="ScrollRight(int)"/>
        protected void ScrollPageRight(int pages)
        {
            this.ScrollRight(pages * this.HorizontalScroll.LargeChange);
        }

        /// <summary>
        /// Modifies current scrollbar properties.
        /// </summary>
        /// <remarks>
        /// The property <i>AutoScrollMinSize</i> is adjusted to currently used 
        /// scaled image size. Additionally, a horizontal and vertical padding 
        /// is also added. Further, the property <i>AutoScrollPosition</i> is 
        /// set to x=0 and y=0 if given parameter <i>reset</i> is true.
        /// </remarks>
        /// <param name="reset">
        /// If true then current scrollbar position is reset.
        /// </param>
        /// <seealso cref="ScaledImageSize"/>
        private void UpdateScrollbars(bool reset)
        {
            this.BeginUpdate();
            try
            {
                if (this.suppressScrollbars)
                {
                    // Just hide all scrollbars 
                    // if they are suppressed.
                    this.VScroll = false;
                    this.HScroll = false;
                    return;
                }

                // Reset current scrolling position if requested.
                if (reset) { this.AutoScrollPosition = new Point(0, 0); }

                // Calculate new auto-scroll area.
                Size size = this.ScaledImageSize.ToSize();
                size.Width += this.Padding.Horizontal;
                size.Height += this.Padding.Vertical;
                this.AutoScrollMinSize = size;
            }
            finally
            {
                this.EndUpdate();
            }
        }

        #endregion // Scrolling related member function implementation.

        #region Other inherited event handler reimplementation.

        /// <summary>
        /// Raises the <i>PaddingChanged</i> event.
        /// </summary>
        /// <remarks>
        /// Additionally, the scrollbars are updated too.
        /// </remarks>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        /// <seealso cref="UpdateScrollbars"/>
        protected override void OnPaddingChanged(EventArgs args)
        {
            // Call as first to ensure PaddingChanged event.
            base.OnPaddingChanged(args);

            // Update scrollbars because they depend on padding (auto invalidate).
            this.UpdateScrollbars(false);
        }

        /// <summary>
        /// Raises the <i>EnabledChanged</i> event.
        /// </summary>
        /// <remarks>
        /// Additionally, the entire control is redrawn.
        /// </remarks>
        /// <param name="args">
        /// A value containing the event data.
        /// </param>
        protected override void OnEnabledChanged(EventArgs args)
        {
            // Call as first to ensure EnabledChanged event.
            base.OnEnabledChanged(args);

            // Redraw entire control.
            this.Invalidate();
        }

        #endregion // Other inherited event handler reimplementation.

        #region Internal property implementation.

        /// <summary>
        /// Gets scaled image size that in fact depends on current zoom 
        /// factor.
        /// </summary>
        /// <remarks>
        /// An empty size is returned if no image is currently available!
        /// </remarks>
        /// <value>
        /// A value representing the image's scaled size.
        /// </value>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ScaledImageOffset"/>
        private SizeF ScaledImageSize
        {
            get
            {
                SizeF result = SizeF.Empty;
                if (this.image != null)
                {
                    result = new SizeF(
                        (float)this.image.Width * (float)this.zoomFactor / 100f,
                        (float)this.image.Height * (float)this.zoomFactor / 100f);
                }
                return result;
            }
        }

        /// <summary>
        /// Gets scaled image offset that in fact depends on current zoom 
        /// factor.
        /// </summary>
        /// <remarks>
        /// An empty point is returned if no image is currently available!
        /// </remarks>
        /// <value>
        /// A value representing the image's scaled offset.
        /// </value>
        /// <seealso cref="ZoomFactor"/>
        /// <seealso cref="ScaledImageSize"/>
        private PointF ScaledImageOffset
        {
            get
            {
                PointF result = PointF.Empty;
                if (this.image != null)
                {
                    float xOffset = 0;
                    float yOffset = 0;
                    float factor = (float)this.zoomFactor / 100f;

                    // Determine horizontal position.
                    if (this.HScroll)
                    {
                        // Horizontal position depends on current scroll position.
                        xOffset = (float)this.AutoScrollPosition.X + (float)this.Padding.Left;
                    }
                    else
                    {
                        // Horizontal position is centered.
                        xOffset = ((float)this.ClientSize.Width - (float)this.image.Width * factor) / 2f;
                    }

                    // Determine vertical position.
                    if (this.VScroll)
                    {
                        // Vertical position depends on current scroll position.
                        yOffset = (float)this.AutoScrollPosition.Y + (float)this.Padding.Top;
                    }
                    else
                    {
                        // Vertical position is centered.
                        yOffset = ((float)this.ClientSize.Height - (float)this.image.Height * factor) / 2f;
                    }

                    result = new PointF(xOffset, yOffset);
                }
                return result;
            }
        }

        #endregion // Internal property implementation.

        #region Win32 API access declarations.

        /// <summary>
        /// Windows message identifier to enable or disable redrawing.
        /// </summary>
        /// <seealso cref="BeginUpdate"/>
        /// <seealso cref="EndUpdate"/>
        /// <seealso cref="SendMessage"/>
        private const int WM_SETREDRAW = 0x000B;

        /// <summary>
        /// Sends specified message to given window and does not return 
        /// until the window procedure has processed the message. 
        /// </summary>
        /// <param name="hWnd">
        /// Handle to the window whose window procedure will receive the 
        /// sent message.
        /// </param>
        /// <param name="nMessage">
        /// Specifies the message identifier to be sent. 
        /// </param>
        /// <param name="wParam">
        /// Specifies additional and message specific information. 
        /// </param>
        /// <param name="lParam">
        /// Specifies additional and message specific information. 
        /// </param>
        /// <returns>
        /// Returns the message processing result and depends on the sent 
        /// message. 
        /// </returns>
        [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Auto)]
        private extern static IntPtr SendMessage(IntPtr hWnd, int nMessage, IntPtr wParam, IntPtr lParam);

        #endregion // Win32 API access declarations.
    }
}
