﻿/*
 * Copyright (C)  2013  Axel Kesseler
 * 
 * This software is free and you can use it for any purpose. Furthermore, 
 * you are free to copy, to modify and/or to redistribute this software.
 * 
 * In addition, this software is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

using System;
using System.IO;
using System.Drawing;
using System.Diagnostics;
using System.Windows.Forms;
using System.ComponentModel;
using System.Runtime.InteropServices;

using plexdata.Utilities;

// A nice article about a complete implementation of an owner drawn list box can 
// be found under http://msdn.microsoft.com/de-de/library/ms229679(v=vs.80).aspx

namespace plexdata.Controls
{
    // NOTE: Adjusting of current column width might be useful! 
    //       In a multi-column environment it could be a good idea 
    //       to adjust current column width. For this purpose just
    //       modify value of property this.ColumnWidth accordingly.
    public class FileListBox : ListBox
    {
        // Define a custom drawing state to be able to distinguish between 
        // internal and external callers of method OnDrawItem().
        private const DrawItemState OwnerPaint = (DrawItemState)0x1000000;

        private bool resizing = false;

        public FileListBox()
            : base()
        {
            this.SetStyle(ControlStyles.UserPaint, true);
            this.SetStyle(ControlStyles.ResizeRedraw, true);
            this.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
            this.SetStyle(ControlStyles.OptimizedDoubleBuffer, true);
            this.UpdateStyles();

            this.ItemHeight = FileListItem.LineCount * this.Font.Height + FileListItem.Padding.Vertical;
            this.DrawMode = DrawMode.OwnerDrawFixed;
        }

        [DefaultValue(50)]
        public override int ItemHeight
        {
            get
            {
                return base.ItemHeight;
            }
            set
            {
                base.ItemHeight = value;
            }
        }

        [DefaultValue(DrawMode.OwnerDrawFixed)]
        public override DrawMode DrawMode
        {
            get
            {
                return base.DrawMode;
            }
            set
            {
                base.DrawMode = value;
            }
        }

        protected override void OnFontChanged(EventArgs args)
        {
            base.OnFontChanged(args);
            this.ItemHeight = FileListItem.LineCount * this.Font.Height + FileListItem.Padding.Vertical;
        }

        protected override void OnPaint(PaintEventArgs args)
        {
            // The core of function OnPaint() comes from page below. Many thanks to that 
            // guy. Further, that page includes some other useful solutions that should 
            // be implemented in the future, such as supporting member DataSource.
            // http://yacsharpblog.blogspot.de/2008/07/listbox-flicker.html

            Region client = new Region(args.ClipRectangle);
            using (Brush brush = new SolidBrush(this.BackColor))
            {
                args.Graphics.FillRegion(brush, client);
            }

            for (int index = this.TopIndex; index < this.Items.Count; index++)
            {
                Rectangle bounds = this.GetItemRectangle(index);

                if (args.ClipRectangle.IntersectsWith(bounds))
                {
                    DrawItemState state = DrawItemState.None | FileListBox.OwnerPaint;

                    if ((this.SelectionMode == SelectionMode.One && this.SelectedIndex == index) ||
                        (this.SelectionMode == SelectionMode.MultiSimple && this.SelectedIndices.Contains(index)) ||
                        (this.SelectionMode == SelectionMode.MultiExtended && this.SelectedIndices.Contains(index)))
                    {
                        state = DrawItemState.Selected;
                    }
                    else
                    {
                        state = DrawItemState.Default;
                    }

                    this.OnDrawItem(new DrawItemEventArgs(
                        args.Graphics, this.Font, bounds, index,
                        state, this.ForeColor, this.BackColor));
                }
            }

            base.OnPaint(args);
        }

        protected override void OnDrawItem(DrawItemEventArgs args)
        {
            // Skip drawing of current item as soon as this method 
            // is called by OnPaint() and resizing is in progress.
            if ((args.State & FileListBox.OwnerPaint) == 0 && this.resizing)
            {
                return;
            }

            Color foreground = this.ForeColor;
            Color background = this.BackColor;

            // Choose the right colors.
            if (this.SelectionMode != SelectionMode.None)
            {
                foreground = args.ForeColor;

                if ((args.State & DrawItemState.Selected) == DrawItemState.Selected)
                {
                    background = args.BackColor;
                }
            }

            // Draw current background. Method DrawBackground() of given 
            // instance of class DrawItemEventArgs is not used by intention.
            using (Brush brush = new SolidBrush(background))
            {
                args.Graphics.FillRectangle(brush, args.Bounds);
            }

            // Draw focus rectangle, if necessary. Method DrawFocusRectangle() of 
            // given instance of class DrawItemEventArgs is not used by intention.
            if ((args.State & DrawItemState.Focus) == DrawItemState.Focus)
            {
                ControlPaint.DrawFocusRectangle(args.Graphics, args.Bounds, foreground, background);
            }

            // Draw current item, if possible.
            if (args.Index >= 0 && args.Index < this.Items.Count && this.Items[args.Index] != null)
            {
                Rectangle bounds = args.Bounds;
                bounds.X += FileListItem.Padding.Left;
                bounds.Y += FileListItem.Padding.Top;
                bounds.Width -= FileListItem.Padding.Horizontal;
                bounds.Height -= FileListItem.Padding.Vertical;

                if (this.Items[args.Index] is FileListItem)
                {
                    (this.Items[args.Index] as FileListItem).DrawItem(
                        args.Graphics, bounds, args.Font, foreground, background);
                }
                else
                {
                    TextRenderer.DrawText(args.Graphics, this.Items[args.Index].ToString(),
                        args.Font, bounds, foreground, background, TextFormatFlags.EndEllipsis);
                }
            }

            base.OnDrawItem(args);
        }

        protected override void WndProc(ref Message message)
        {
            const int WM_SIZE = 0x05;

            if (message.Msg == WM_SIZE)
            {
                this.resizing = true;
                base.WndProc(ref message);
                this.resizing = false;
            }
            else
            {
                base.WndProc(ref message);
            }
        }
    }

    public class FileListItem
    {
        public static int LineCount = 4;
        public static Padding Padding = new Padding(1, 1, 2, 2);

        public FileListItem()
            : base()
        {
            this.FileInfo = null;
            this.SmallIcon = null;
            this.LargeIcon = null;
        }

        public FileListItem(string filename)
            : this()
        {
            if (!String.IsNullOrEmpty(filename))
            {
                try
                {
                    this.FileInfo = new FileInfo(filename);
                    this.SmallIcon = FileListItem.LoadIcon(filename, true);
                    this.LargeIcon = FileListItem.LoadIcon(filename, false);
                }
                catch (Exception exception)
                {
                    Debug.WriteLine(exception);
                }
            }
        }

        public FileInfo FileInfo { get; private set; }

        public string FullName
        {
            get
            {
                if (this.FileInfo != null)
                {
                    return this.FileInfo.FullName;
                }
                else
                {
                    return String.Empty;
                }
            }
        }

        public string FileName
        {
            get
            {
                if (this.FileInfo != null)
                {
                    return this.FileInfo.Name;
                }
                else
                {
                    return String.Empty;
                }
            }
        }

        public string FilePath
        {
            get
            {
                if (this.FileInfo != null)
                {
                    return this.FileInfo.DirectoryName;
                }
                else
                {
                    return String.Empty;
                }
            }
        }

        public Icon SmallIcon { get; private set; }

        public Icon LargeIcon { get; private set; }

        public override string ToString()
        {
            if (this.FileInfo != null)
            {
                return this.FileInfo.FullName;
            }
            else
            {
                return base.ToString();
            }
        }

        public void DrawItem(Graphics graphics, Rectangle bounds, Font font, Color foreground, Color background)
        {
            if (this.FileInfo != null)
            {
                if (this.LargeIcon != null)
                {
                    int w = this.LargeIcon.Width;
                    int h = this.LargeIcon.Height;
                    int x = bounds.Left;
                    int y = bounds.Top + (bounds.Height - h) / 2;

                    graphics.DrawIconUnstretched(this.LargeIcon, new Rectangle(x, y, w, h));

                    // Use a small gap between icon and the rest.
                    bounds.X += w + 5;
                    bounds.Width -= w + 5;
                }

                bounds.Height = font.Height;

                TextRenderer.DrawText(
                    graphics, this.FileName, font, bounds,
                    foreground, background, TextFormatFlags.EndEllipsis);

                foreground = ControlPaint.LightLight(foreground);

                bounds.Y += font.Height;

                TextRenderer.DrawText(
                    graphics, this.FilePath, font, bounds,
                    foreground, background, TextFormatFlags.EndEllipsis);

                bounds.Y += font.Height;

                TextRenderer.DrawText(
                    graphics, this.FileInfo.CreationTime.ToString(), font, bounds,
                    foreground, background, TextFormatFlags.EndEllipsis);

                bounds.Y += font.Height;

                TextRenderer.DrawText(
                    graphics, CapacityConverter.Convert(this.FileInfo.Length), font, bounds,
                    foreground, background, TextFormatFlags.EndEllipsis);
            }
        }

        private static Icon LoadIcon(string filename, bool small)
        {
            const int FILE_ATTRIBUTE_NORMAL = 0x00000080;
            const int SHGFI_ICON = 0x100;
            const int SHGFI_LARGEICON = 0x0;
            const int SHGFI_SMALLICON = 0x1;
            const int SHGFI_USEFILEATTRIBUTES = 0x10;

            Icon result = null;
            SHFILEINFO info = new SHFILEINFO();
            try
            {
                int size = Marshal.SizeOf(info);
                int attr = FILE_ATTRIBUTE_NORMAL;
                int flags = SHGFI_USEFILEATTRIBUTES | SHGFI_ICON | (small ? SHGFI_SMALLICON : SHGFI_LARGEICON);

                // Use wildcard in case given file does not exist.
                filename = "*" + Path.GetExtension(filename);

                SHGetFileInfo(filename, attr, ref info, size, flags);

                int error = Marshal.GetLastWin32Error();

                result = (Icon)(Icon.FromHandle(info.hIcon).Clone());

                if (result == null)
                {
                    throw new Win32Exception(error);
                }
                else if (result.Size == Size.Empty)
                {
                    if (error == 0) { error = Marshal.GetLastWin32Error(); }

                    throw new Win32Exception(error);
                }
            }
            finally
            {
                if (info.hIcon != IntPtr.Zero) { DestroyIcon(info.hIcon); }
            }

            return result;
        }

        #region Win32 API function declarations.

        [StructLayout(LayoutKind.Sequential)]
        private struct SHFILEINFO
        {
            public IntPtr hIcon;
            public IntPtr nIcon;
            public uint attributes;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 260)] // MAX_PATH
            public string displayName;
            [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 80)]
            public string typeName;
        };

        [DllImport("shell32.dll", CallingConvention = CallingConvention.StdCall, SetLastError = true, PreserveSig = true, CharSet = CharSet.Unicode)]
        private static extern IntPtr SHGetFileInfo(string fullpath, int fileAttributes, ref SHFILEINFO fileInfo, int sizeFileInfo, int flags);

        [DllImport("user32.dll", CallingConvention = CallingConvention.Winapi, SetLastError = true, PreserveSig = true)]
        private static extern bool DestroyIcon(IntPtr hIcon);

        #endregion // Win32 API function declarations.
    }
}
